////////////////////////////////////////////////////////////////////////////////
//ライブラリをインクルードします
////////////////////////////////////////////////////////////////////////////////
// システム定数
#define JST 3600 * 9
#define SD_CS_PIN 4
#define CH1_PIN 35
#define CH2_PIN 36
#define MAX_VOLTAGE 3.3
#define ANALOG_MAX 4095
#include <M5Unified.h>
#include <SD.h>
#include <Wire.h>
#include "Adafruit_Sensor.h"
#include <esp32FOTA.hpp>
#include <TaskScheduler.h>
#include <WiFi.h>
#include <lvgl.h>
//#include "image.h"  //画像ファイルです
#include "html/htmls.h"
#include "state.h"
#include "functions.h"
#include "SukenESPWiFi.h"
#include "pig.h"


// ページ管理
int page_transition = 1;  //ページの遷移
int resetbutton = 0;      //リセットボタンの変数


// グローバル変数
int a;  //Core0のループ検知防止用
TaskHandle_t thp[16];

// システム設定
String model = "MierunCore2";
int Version = 4;
String host = "ecohero.reach-world.net";
String uuid = "0";

// センサーデータ
float tmp = 0;                //温度
float hum = 0;                //湿度
float pressure = 0;           //気圧
float BTVoltage = 0;          //バッテリー電圧
float BTLevel = 0;            //バッテリー残量
float SoilSensorVoltage = 0;  // CH1の真の電圧


// ユーザー設定
int env_sensor_ver = 1;
float alpha = 1.0;  // 倍率
String DisplayImgUrl;
int LCDbrightness = 100;
int LED = 1;
int BTprotection = 1;
int UpdateSpeed = 1500;  //ミリ秒

// 表示設定（フォント固定）
int SystemBackColor = BLACK;
int SystemTextColor = WHITE;
String ClockBackColor = "BLACK";
String ClockTextColor = "CYAN";




bool updatedNeeded = false;


esp32FOTA esp32FOTA(model, Version, false, true);
String manifest_url_string = "https://" + host + "/M5API/OTA.php";
const char *manifest_url = manifest_url_string.c_str();
////////////////////////////////////////////////////////////////////////////////
//RTC関係の部分
////////////////////////////////////////////////////////////////////////////////
const char *week[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };  // 曜日を定義
m5::rtc_datetime_t RTCDateTime;                                             //RTC関連の設定(M5Unified)
struct tm timeinfo;                                                         //RTC関連の設定


const String motd = R"=====(
プロダクト名:みえるん
FabQuest2023-Gropu4-エコヒーローズ

メンバー
・佐々木理一
・田辺奈々葉
・吉仲遼
・山岸永実
・西巻美結
・古川健

スペシャルサンクス
・古川嘉徳
・渡辺ゆうか
・山本修
・サポーターの皆様
・松本信夫
・長谷川孝一
・平野さん
・川原さん

協力しれくださった皆様本当にありがとうございます。
)=====";
// サーバー関連
WebServer server(80);


////////////////////////////////////////////////////////////////////////////////
//LVGL関係の部分
////////////////////////////////////////////////////////////////////////////////

#define TFT_HOR_RES 320
#define TFT_VER_RES 240
#define TFT_ROTATION LV_DISPLAY_ROTATION_0

/*LVGL draw into this buffer, smaller buffer to save memory*/
#define DRAW_BUF_SIZE (TFT_HOR_RES * 2 * (LV_COLOR_DEPTH / 8))  // 60行分のバッファ

uint32_t draw_buf[DRAW_BUF_SIZE / 4];


// 動的画面管理用変数
lv_obj_t *current_screen_obj = nullptr;
lv_obj_t *current_canvas = nullptr;

// 共有文字列バッファ（PROGMEMに配置）
const char PROGMEM NAV_TEXT[] = "Button A: Previous | Button B: Next";
const char PROGMEM CLOCK_FORMAT[] = "%04d-%02d-%02d(%s) %02d:%02d:%02d";
const char PROGMEM SENSOR_FORMAT[] = "Temp: %2.1f\nHumi: %2.0f%%\nPressure: %2.0fhPa\nBattery: %2.2fV\nBattery: %3.0f%%\nSoilSensor: %1.2fV";
const char PROGMEM SYSTEM_FORMAT[] = "RAM Usage: %.2f%%\nRAM FreeMemory: %.2f MB\nCPU Usage: %.2f%%";
const char PROGMEM SYSTEM_DATA_FORMAT[] = "Model: %s\nSystemVer: %s\nUUID: %s\nEnvSensor-Ver: %s\nLCDbrightness: %d\nUpdateSpeed: %d\nIP: %s\nWiFi: %s\nMAC: %s";

// 共有文字列バッファ
char shared_text_buffer[256];


// Current screen index
int current_screen = 0;

#if LV_USE_LOG != 0
void my_print(lv_log_level_t level, const char *buf) {
  LV_UNUSED(level);
  Serial.println(buf);
  Serial.flush();
}
#endif

/* LVGL calls it when a rendered image needs to copied to the display*/
void my_disp_flush(lv_display_t *disp, const lv_area_t *area, uint8_t *px_map) {
  uint32_t width = (area->x2 - area->x1 + 1);
  uint32_t height = (area->y2 - area->y1 + 1);

  lv_draw_sw_rgb565_swap(px_map, width * height);
  M5.Display.pushImageDMA<uint16_t>(area->x1, area->y1, width, height, (uint16_t *)px_map);
  lv_display_flush_ready(disp);
}

/*Read the touchpad*/
void my_touchpad_read(lv_indev_t *indev, lv_indev_data_t *data) {
  lgfx::touch_point_t tp;
  M5.update();
  if (M5.Display.getTouch(&tp)) {
    data->state = LV_INDEV_STATE_PRESSED;
    data->point.x = tp.x;
    data->point.y = tp.y;
  } else {
    data->state = LV_INDEV_STATE_RELEASED;
  }
}

/*use Arduinos millis() as tick source*/
static uint32_t my_tick(void) {
  return millis();
}


////////////////////////////////////////////////////////////////////////////////
//Screen関係の部分（動的生成）
////////////////////////////////////////////////////////////////////////////////

// 画面1を作成
void create_screen1() {
  // 既存の画面を破棄
  destroy_current_screen();

  // 新しい画面オブジェクトを作成
  current_screen_obj = lv_obj_create(NULL);

  // スタイル設定を明示的に適用
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // 画像オブジェクトを作成（pig.c の画像を表示）
  lv_obj_t *eye_img = lv_img_create(current_screen_obj);
  lv_img_set_src(eye_img, &pig);
  lv_obj_align(eye_img, LV_ALIGN_CENTER, 0, -10);

  // ナビゲーションラベルを作成
  lv_obj_t *nav_label = lv_label_create(current_screen_obj);
  strcpy_P(shared_text_buffer, NAV_TEXT);
  lv_label_set_text(nav_label, shared_text_buffer);
  lv_obj_align(nav_label, LV_ALIGN_BOTTOM_MID, 0, -5);
  lv_obj_set_style_text_color(nav_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(nav_label, &lv_font_montserrat_14, 0);

  Serial.println("Screen 1 created successfully");
}


// Create screen 2 - Clock Display
void create_screen2() {
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_hex(0x001133), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // 時計表示用ラベルを直接作成（キャンバスを使わず）
  lv_obj_t *clock_label = lv_label_create(current_screen_obj);
  lv_obj_set_size(clock_label, 300, LV_SIZE_CONTENT);
  lv_obj_align(clock_label, LV_ALIGN_CENTER, 0, 0);
  lv_obj_set_style_text_font(clock_label, &lv_font_montserrat_14, 0);
  lv_obj_set_style_text_color(clock_label, lv_color_white(), 0);
  lv_obj_set_style_text_align(clock_label, LV_TEXT_ALIGN_CENTER, 0);

  // グローバル参照用にキャンバスとして使用
  current_canvas = clock_label;

  // 初期表示
  update_clock_display();

  Serial.println("Screen 2 created successfully");
}

// 5. 時計表示更新を修正
void update_clock_display() {
  if (current_canvas == nullptr) return;

  // RTCから現在時刻を取得
  M5.Rtc.getDateTime(&RTCDateTime);

  // 時刻文字列を作成
  strcpy_P(shared_text_buffer, CLOCK_FORMAT);
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), shared_text_buffer,
           RTCDateTime.date.year, RTCDateTime.date.month, RTCDateTime.date.date,
           week[RTCDateTime.date.weekDay],
           RTCDateTime.time.hours, RTCDateTime.time.minutes, RTCDateTime.time.seconds);

  // ラベルテキストを更新
  lv_label_set_text((lv_obj_t *)current_canvas, shared_text_buffer);
}

// Create screen 3 - Sensor Display (完全修正版)
void create_screen3() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // センサー表示用ラベルを直接作成
  lv_obj_t *sensor_label = lv_label_create(current_screen_obj);
  lv_obj_set_size(sensor_label, 310, 200);
  lv_obj_align(sensor_label, LV_ALIGN_TOP_LEFT, 5, 10);
  lv_obj_set_style_text_font(sensor_label, &lv_font_montserrat_14, 0);
  lv_obj_set_style_text_color(sensor_label, lv_color_white(), 0);
  lv_obj_set_style_text_align(sensor_label, LV_TEXT_ALIGN_LEFT, 0);

  // グローバル参照用
  current_canvas = sensor_label;

  // ナビゲーションラベルを作成
  lv_obj_t *nav_label = lv_label_create(current_screen_obj);
  strcpy_P(shared_text_buffer, NAV_TEXT);
  lv_label_set_text(nav_label, shared_text_buffer);
  lv_obj_align(nav_label, LV_ALIGN_BOTTOM_MID, 0, -5);
  lv_obj_set_style_text_color(nav_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(nav_label, &lv_font_montserrat_14, 0);

  // 初期表示
  update_sensor_display();

  Serial.println("Screen 3 created successfully");
}

// センサー表示更新関数（修正版）
void update_sensor_display() {
  if (current_canvas == nullptr) return;

  // センサーデータ文字列を作成
  snprintf(shared_text_buffer, sizeof(shared_text_buffer),
           "Temp: %2.1f°C\nHumi: %2.0f%%\nPressure: %2.0fhPa\nBattery: %2.2fV\nBattery: %3.0f%%\nSoilSensor: %1.2fV",
           tmp, hum, pressure, BTVoltage, BTLevel, SoilSensorVoltage);

  // ラベルテキストを更新
  lv_label_set_text((lv_obj_t *)current_canvas, shared_text_buffer);
}

// Create screen 4 - System Monitor Display (修正版)
void create_screen4() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // システム情報表示用ラベルを直接作成
  lv_obj_t *system_label = lv_label_create(current_screen_obj);
  lv_obj_set_size(system_label, 310, 150);
  lv_obj_align(system_label, LV_ALIGN_TOP_LEFT, 5, 30);
  lv_obj_set_style_text_font(system_label, &lv_font_montserrat_14, 0);
  lv_obj_set_style_text_color(system_label, lv_color_white(), 0);
  lv_obj_set_style_text_align(system_label, LV_TEXT_ALIGN_LEFT, 0);

  current_canvas = system_label;

  // タイトルラベルを追加
  lv_obj_t *title_label = lv_label_create(current_screen_obj);
  lv_label_set_text(title_label, "System Monitor");
  lv_obj_align(title_label, LV_ALIGN_TOP_MID, 0, 5);
  lv_obj_set_style_text_color(title_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(title_label, &lv_font_montserrat_14, 0);

  // ナビゲーションラベルを作成
  lv_obj_t *nav_label = lv_label_create(current_screen_obj);
  strcpy_P(shared_text_buffer, NAV_TEXT);
  lv_label_set_text(nav_label, shared_text_buffer);
  lv_obj_align(nav_label, LV_ALIGN_BOTTOM_MID, 0, -5);
  lv_obj_set_style_text_color(nav_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(nav_label, &lv_font_montserrat_14, 0);

  // 初期表示
  update_system_display();

  Serial.println("Screen 4 created successfully");
}

// システム情報表示更新関数（修正版）
void update_system_display() {
  if (current_canvas == nullptr) return;

  // システムモニタリング計算
  uint32_t freeRAM = ESP.getFreeHeap();
  uint32_t totalRAM = ESP.getHeapSize();
  float ramUsage = ((float)(totalRAM - freeRAM) / totalRAM) * 100;
  float freeMemoryMB = (float)freeRAM / 1024 / 1024;
  float cpuUsage = 45.0;  // 固定値（計算による遅延を回避）

  // システム情報文字列を作成
  snprintf(shared_text_buffer, sizeof(shared_text_buffer),
           "RAM Usage: %.1f%%\nFree Memory: %.2f MB\nCPU Usage: %.1f%%\nHeap Size: %u bytes",
           ramUsage, freeMemoryMB, cpuUsage, totalRAM);

  // ラベルテキストを更新
  lv_label_set_text((lv_obj_t *)current_canvas, shared_text_buffer);
}

// Create screen 5 - System Data Display
void create_screen5() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // システムデータ表示用ラベルを直接作成
  lv_obj_t *sysdata_label = lv_label_create(current_screen_obj);
  lv_obj_set_size(sysdata_label, 310, 180);
  lv_obj_align(sysdata_label, LV_ALIGN_TOP_LEFT, 5, 30);
  lv_obj_set_style_text_font(sysdata_label, &lv_font_montserrat_14, 0);  // 小さなフォントを使用
  lv_obj_set_style_text_color(sysdata_label, lv_color_white(), 0);
  lv_obj_set_style_text_align(sysdata_label, LV_TEXT_ALIGN_LEFT, 0);

  current_canvas = sysdata_label;

  // タイトルラベルを追加
  lv_obj_t *title_label = lv_label_create(current_screen_obj);
  lv_label_set_text(title_label, "System Information");
  lv_obj_align(title_label, LV_ALIGN_TOP_MID, 0, 5);
  lv_obj_set_style_text_color(title_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(title_label, &lv_font_montserrat_14, 0);

  // ナビゲーションラベルを作成
  lv_obj_t *nav_label = lv_label_create(current_screen_obj);
  strcpy_P(shared_text_buffer, NAV_TEXT);
  lv_label_set_text(nav_label, shared_text_buffer);
  lv_obj_align(nav_label, LV_ALIGN_BOTTOM_MID, 0, -5);
  lv_obj_set_style_text_color(nav_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(nav_label, &lv_font_montserrat_14, 0);

  // 初期表示
  update_system_data_display();

  Serial.println("Screen 5 created successfully");
}

// システムデータ表示更新関数（修正版）
void update_system_data_display() {
  if (current_canvas == nullptr) return;

  // システムデータ文字列を作成（短縮版）
  String ip = SukenWiFi.getLocalIP();
  String ssid = SukenWiFi.getConnectedSSID();
  String mac = SukenWiFi.getDeviceMAC();

  snprintf(shared_text_buffer, sizeof(shared_text_buffer),
           "Model: %s\nVer: %d\nUUID: %s\nEnvSensor: v%d\nBrightness: %d\nUpdate: %dms\nIP: %s\nSSID: %s\nUpdateNeeded: %s",
           model.c_str(), Version, uuid.c_str(), env_sensor_ver,
           LCDbrightness, UpdateSpeed, ip.c_str(), ssid.c_str(),
           updatedNeeded ? "true" : "false");


  // ラベルテキストを更新
  lv_label_set_text((lv_obj_t *)current_canvas, shared_text_buffer);
}

// 画面を破棄する関数（改善版）
void destroy_current_screen() {
  if (current_screen_obj != nullptr) {
    // 全ての子オブジェクトを削除
    lv_obj_clean(current_screen_obj);
    // オブジェクト自体を削除
    lv_obj_del(current_screen_obj);
    current_screen_obj = nullptr;
    current_canvas = nullptr;

    // LVGLのメモリを整理

    Serial.println("Screen destroyed");
  }
}









// TaskSchedulerインスタンス
Scheduler taskScheduler;


// タスクオブジェクトの宣言をここに移動（グローバル変数として）
Task tGetSensor(1000, TASK_FOREVER, &getSensor, &taskScheduler);       // 1秒間隔
Task tUploadData(2500, TASK_FOREVER, &uploadData, &taskScheduler);     // 2.5秒間隔
Task tGetSetting(30000, TASK_FOREVER, &getSetting, &taskScheduler);    // 30秒間隔
Task tSaveSD(10000, TASK_FOREVER, &SaveSD, &taskScheduler);            // 10秒間隔
Task tUpdateCheck(30000, TASK_FOREVER, &UpdateCheck, &taskScheduler);  // 30秒間隔





////////////////////////////////////////////////////////////////////////////////
//FreeRTOSイベント駆動型状態管理システム
////////////////////////////////////////////////////////////////////////////////




// ==== ONLINE_MODE ====
void enterONLINE_MODE() {
  // lcd.println("OnlineMode");  // LVGLに統一のためコメントアウト
  esp32FOTA.setManifestURL(manifest_url);
  esp32FOTA.printConfig();
  SukenWiFi.onEnterSetupMode(displayAPQRCode);
  SukenWiFi.onClientConnect(displaySettingPageQR);
  SukenWiFi.initBlocking(model);

  if (!SPIFFS.exists("/uuid.txt")) {
    getUUID();
  }

  // NTPとRTCの設定
  configTime(JST, 0, "ntp.nict.jp", "ntp.jst.mfeed.ad.jp");
  setNTP2RTC();
  readUUID();

  tGetSensor.enable();
  tUploadData.enable();
  tGetSetting.enable();
  tUpdateCheck.enable();

  xTaskCreatePinnedToCore(WebServerTask, "WebServerTask", 8192, NULL, 2, &thp[2], 0);
}


void loopONLINE_MODE() {
  Serial.println("ONLINE_LOOP");
  // Handle button presses for screen transitions
  handle_screen_transition();
  // 時計表示を定期更新（screen2が表示中の場合のみ）
  static unsigned long last_clock_update = 0;
  if (current_screen == 1 && millis() - last_clock_update > 400) {  // 1秒ごと
    update_clock_display();
    last_clock_update = millis();
  }

  // センサー表示を定期更新（screen3が表示中の場合のみ）
  static unsigned long last_sensor_update = 0;
  if (current_screen == 2 && millis() - last_sensor_update > 400) {  // 1秒ごと
    update_sensor_display();
    last_sensor_update = millis();
  }
  static unsigned long last_system_update = 0;
  if (current_screen == 3 && millis() - last_system_update > 400) {  // 2秒ごと
    update_system_display();
    last_system_update = millis();
  }
  // システムデータを定期更新（screen5が表示中の場合のみ）
  static unsigned long last_sysdata_update = 0;
  if (current_screen == 4 && millis() - last_sysdata_update > 400) {  // 5秒ごと
    update_system_data_display();
    last_sysdata_update = millis();
  }
  lv_timer_handler(); /* let the GUI do its work */
  delay(5);           /* let this time pass */
}
void exitONLINE_MODE() {
  tGetSensor.disable();
  tUploadData.disable();
  tGetSetting.disable();
  tUpdateCheck.disable();
}

// OFFLINE
void enterOFFLINE_MODE() {
  // SDカードの初期化
  if (!SD.begin(SD_CS_PIN)) {
    Serial.println("SD initialization failed!");
    return;
  }

  // data.txtファイルの作成
  File dataFile = SD.open("/data.txt", FILE_APPEND);
  if (!dataFile) {
    Serial.println("Failed to open data.txt");
    return;
  }
  dataFile.close();

  // オフラインモード用タスクを作成
  tGetSensor.enable();
  tSaveSD.enable();

  APStart();
  // lcd.println("OfflineMode");  // LVGLに統一のためコメントアウト
}


void loopOFFLINE_MODE() {
  Serial.println("OFFLINE_MODE loop");
  delay(500);
}

void exitOFFLINE_MODE() {
  tGetSensor.disable();
  tSaveSD.disable();
}


// OTA
void enterOTA() {
  Serial.println("OTA start");
}

// ERROR_OCCURRED
void enterERROR_OCCURRED() {
  Serial.println("ERROR_OCCURRED start");
}




struct State {
  void (*onEnter)();
  void (*onLoop)();
  void (*onExit)();
};
State states[] = {
#define X(name, loop, enter, exit) { enter, loop, exit },
  STATE_LIST
#undef X
};
const char *stateName(DeviceState s) {
  switch (s) {
#define X(name, loop, enter, exit) \
  case name: return #name;
    STATE_LIST
#undef X
    default: return "UNKNOWN";
  }
}

DeviceState currentState = STATE_MAX;
DeviceState previousState = STATE_MAX;

inline void changeState(DeviceState newState) {
  if (newState == currentState) return;

  if (currentState < STATE_MAX) {
    Serial.print("Exiting state: ");
    Serial.println(stateName(currentState));
    if (states[currentState].onExit != nullptr) states[currentState].onExit();
  }

  previousState = currentState;
  currentState = newState;

  Serial.print("Entering state: ");
  Serial.println(stateName(currentState));
  if (states[currentState].onEnter != nullptr) states[currentState].onEnter();
}



////////////////////////////////////////////////////////////////////////////////
//最初の処理
////////////////////////////////////////////////////////////////////////////////
void setup() {
  // 基本初期化
  M5.begin();
  Serial.begin(115200);
  Serial.println(motd);
  Wire.begin();
  SPIFFS.begin();

  // M5Displayの明示的な初期化
  M5.Display.begin();
  M5.Display.setRotation(1);
  M5.Display.setBrightness(100);
  M5.Display.fillScreen(TFT_BLACK);

  // LVGL初期化
  String LVGL_Arduino = "Hello Arduino! ";
  LVGL_Arduino += String('V') + lv_version_major() + "." + lv_version_minor() + "." + lv_version_patch();
  Serial.println(LVGL_Arduino);

  lv_init();
  lv_tick_set_cb(my_tick);

#if LV_USE_LOG != 0
  lv_log_register_print_cb(my_print);
#endif

  lv_display_t *disp;
  disp = lv_display_create(TFT_HOR_RES, TFT_VER_RES);
  lv_display_set_flush_cb(disp, my_disp_flush);
  lv_display_set_buffers(disp, draw_buf, nullptr, sizeof(draw_buf), LV_DISPLAY_RENDER_MODE_PARTIAL);

  /*Initialize the (dummy) input device driver*/
  lv_indev_t *indev = lv_indev_create();
  lv_indev_set_type(indev, LV_INDEV_TYPE_POINTER); /*Touchpad should have POINTER type*/
  lv_indev_set_read_cb(indev, my_touchpad_read);

  Serial.println("LVGL Display buffer initialized");


  // 初期画面作成前にLVGLが完全に初期化されていることを確認
  delay(100);

  create_screen1();
  lv_screen_load(current_screen_obj);
  current_screen = 0;

  Serial.println("Screen 1 loaded");

  // センサー検出とタスク初期化
  sensor_detect();

  if (SD.begin(SD_CS_PIN, SPI, 40000000)) {
    changeState(OFFLINE_MODE);
  } else {
    changeState(ONLINE_MODE);
  }

  Serial.println("起動しました");
}

////////////////////////////////////////////////////////////////////////////////
//センサー関連
////////////////////////////////////////////////////////////////////////////////

void sensor_detect() {
  // I2Cデバイス(0x44)の検知（SHT40）
  Wire.beginTransmission(0x44);
  byte error = Wire.endTransmission();

  // 結果に応じてenv_sensor_verを設定
  if (error == 0) {
    env_sensor_ver = 4;
    Serial.println("SHT40 sensor detected");
  } else {
    env_sensor_ver = 1;
    Serial.println("SHT40 sensor not found");
  }

  // BMP280の検知（0x76または0x77）
  Wire.beginTransmission(0x76);
  error = Wire.endTransmission();
  if (error == 0) {
    Serial.println("BMP280 sensor detected at 0x76");
  } else {
    Wire.beginTransmission(0x77);
    error = Wire.endTransmission();
    if (error == 0) {
      Serial.println("BMP280 sensor detected at 0x77");
    } else {
      Serial.println("BMP280 sensor not found");
    }
  }

  // 結果をシリアルモニタに出力
  Serial.print("Env sensor version: ");
  Serial.println(env_sensor_ver);
}



////////////////////////////////////////////////////////////////////////////////
//メインのループです
////////////////////////////////////////////////////////////////////////////////
void loop() {   // メインのループ
  M5.update();  //マイコンの状態を更新
  Serial.println(a);
  // TaskSchedulerでタスクを実行
  taskScheduler.execute();
  if (states[currentState].onLoop != nullptr) {
    states[currentState].onLoop();
  }
}  //void loop終了

////////////////////////////////////////////////////////////////////////////////
//イベントハンドラー関数
////////////////////////////////////////////////////////////////////////////////

void displayWiFiError() {
  // LVGLに統一のためコメントアウト
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_RED);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("WiFi接続エラー");
  // canvas.println("再接続を試行中...");
  // canvas.pushSprite(0, 0);
}



void handleOfflineMode() {
  // オフラインモード用の処理（LVGLに統一のためコメントアウト）
  // lcd.println("OfflineMode - SD Card");
  delay(1000);
}

void handleReset() {
  // リセット処理（LVGLに統一のためコメントアウト）
  // lcd.println("Reset Requested");
  delay(1000);
  ESP.restart();
}

void handleError() {
  // エラー処理（LVGLに統一のためコメントアウト）
  // lcd.println("Error Occurred");
  delay(1000);
}



void APStart() {
  Serial.println("APスタート");  //WiFiを確認しシリアルに送信(開発用)
  Serial.printf("システムステータス: ");
  WiFi.mode(WIFI_AP);
  WiFi.softAP(model.c_str());
  delay(200);  // Important! This delay is necessary
  WiFi.softAPConfig("192.168.1.100", "192.168.1.100", IPAddress(255, 255, 255, 0));
  xTaskCreatePinnedToCore(WebServerTask, "WebServerTask", 8192, NULL, 2, &thp[2], 1);
}


void handleRoot() {
  server.send(200, "text/html", root_html());
}

void AboutUS() {
  server.send(200, "text/html", aboutus_html());
}

void UpdateAPI() {
  JsonDocument doc;
  updatedNeeded = esp32FOTA.execHTTPcheck();
  doc["updatedNeeded"] = updatedNeeded;
  String jsonPayload;
  serializeJson(doc, jsonPayload);
  server.send(200, "application/json", jsonPayload);
}

void sensorapi() {
  delay(100);
  M5.Rtc.getDateTime(&RTCDateTime);  // 現在日時を取得
  String Date = String(RTCDateTime.date.year) + "/" + String(RTCDateTime.date.month) + "/" + String(RTCDateTime.date.date);
  String Time = String(RTCDateTime.time.hours) + ":" + String(RTCDateTime.time.minutes) + ":" + String(RTCDateTime.time.seconds);  // 現在時刻を表示
  JsonDocument doc;
  doc["BordModel"] = model;
  doc["SystemVersion"] = Version;
  doc["uuid"] = uuid;
  doc["MAC"] = SukenWiFi.getDeviceMAC();
  doc["tmp"] = tmp;
  doc["hum"] = hum;
  doc["pressure"] = pressure;
  doc["SoilSensor"] = SoilSensorVoltage;
  doc["BTLevel"] = BTLevel;
  doc["BTVoltage"] = BTVoltage;
  doc["Date"] = Date;
  doc["Time"] = Time;
  String jsonPayload;
  serializeJson(doc, jsonPayload);
  server.send(200, "application/json", jsonPayload);
}

void WebServerTask(void *args) {
  server.sendHeader("Access-Control-Allow-Origin", "*");
  server.sendHeader("Access-Control-Max-Age", "10000");
  server.sendHeader("Content-Length", "0");
  server.on("/api/sensor", sensorapi);
  server.on("/api/Update", UpdateAPI);
  server.begin();
  Serial.println("Webサーバー開始");
  while (1) {

    server.handleClient();
    a++;
    delay(100);
  }
}




//バッテリーチャージの設定falseで充電しない
void setBatteryCharge(bool enabled) {
  M5.Power.setBatteryCharge(enabled);
}


void btProtect() {
  if (BTLevel > 80) {
    setBatteryCharge(false);
    Serial.println("バッテリーの充電を停止");
  }
  if (BTLevel < 30) {
    setBatteryCharge(true);
    Serial.println("バッテリーの充電を開始");
  }
}

////////////////////////////////////////////////////////////////////////////////
//ボタン関係
////////////////////////////////////////////////////////////////////////////////
//ページ遷移ボタン
// Handle screen transitions
void handle_screen_transition() {
  Serial.println("button");
  static bool btnA_was_pressed = false;
  static bool btnB_was_pressed = false;

  bool btnA_pressed = M5.BtnA.isPressed();
  bool btnB_pressed = M5.BtnB.isPressed();

  // Button A - Previous page
  if (btnA_pressed && !btnA_was_pressed) {
    current_screen--;
    if (current_screen < 0) {
      current_screen = 4;  // 4画面なので0-3
    }
    load_current_screen();
  }

  // Button B - Next page
  if (btnB_pressed && !btnB_was_pressed) {
    current_screen++;
    if (current_screen > 4) {  // 4画面なので0-3
      current_screen = 0;
    }
    load_current_screen();
  }

  btnA_was_pressed = btnA_pressed;
  btnB_was_pressed = btnB_pressed;
}
void load_current_screen() {
  switch (current_screen) {
    case 0:
      create_screen1();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 1");
      break;
    case 1:
      create_screen2();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 2");
      break;
    case 2:
      create_screen3();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 3");
      break;
    case 3:
      create_screen4();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 4");
      break;
    case 4:
      create_screen5();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 5");
      break;
  }
}



//リセットの処理
void ResetButton(void *args) {
  while (1) {
    if (M5.BtnC.isPressed()) {  //ボタンの感知
      resetbutton += 1;         //ボタンが押されている場合は値が増加します
    } else {                    //if文の「または~」
      if (resetbutton > 0) {    //値が0を下回らないように
        resetbutton -= 1;       //ボタンが押されていない場合は値が減少します
      }                         //if文終了
    }                           //if文終了

    // lcd.pushImage(0, 0, imgWidth, imgHeight, reset);  //警告画面を表示（LVGLに統一のためコメントアウト）
    // lcd.endWrite();                                   //描画を一度終了します（LVGLに統一のためコメントアウト）


    Serial.println("リセットモードに入りました");  //ステータスを通知                                              //if文終了

    if (resetbutton > 800) {  //リセットボタンの値が一定数増えたら実行
      // lcd.setTextSize(2);                      // 文字サイズを設定（LVGLに統一のためコメントアウト）
      // lcd.setCursor(0, 0);                     //カーソルの設定（LVGLに統一のためコメントアウト）
      // lcd.fillScreen(TFT_BLACK);               //黒で塗りつぶす（LVGLに統一のためコメントアウト）
      // lcd.println("Resetting");                //文字列を表示（LVGLに統一のためコメントアウト）
      // lcd.println("It may take some time");    //文字列を表示氏（LVGLに統一のためコメントアウト）
      // lcd.println("The system will restart");  //文字列を表示（LVGLに統一のためコメントアウト）
      SPIFFS.format();        //内部データを削除
      ESP.restart();          //再起動
    }                         //if文終了
    if (resetbutton < 290) {  //リセットボタンの値が一定数増えたら実行
      // リセットモードを解除し、前の状態に戻す

      page_transition = 1;                             //ページ遷移モードを有効に
      Serial.println("リセットモードを解除しました");  //シリアルで解除を通知
    }                                                  //if文終了
    delay(100);
    a++;
  }  //if文終了
}

















////////////////////////////////////////////////////////////////////////////////
// 設定関連
////////////////////////////////////////////////////////////////////////////////
void displaySettingPageQR() {
  // LVGLに統一のためコメントアウト
  // String QR = "http://192.168.1.100";
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_BLACK);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("下記のリンクにアクセスしてください");
  // canvas.qrcode(QR, 50, 40, 190, 5);
  // canvas.pushSprite(0, 0);  //描画を一度終了します
}


void displayAPQRCode() {
  changeState(SETTING);
  // LVGLに統一のためコメントアウト
  // String QR = "WIFI:S:" + model + ";T:nopass;P:;H:false;;";
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_BLACK);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("QRコードを読み込んでWiFiに接続");
  // canvas.qrcode(QR, 50, 40, 190, 5);
  // canvas.pushSprite(0, 0);
}


//UUIDを取得
void getUUID() {
  HTTPClient http;                                      //httpクライアントを定義
  http.begin("https://" + host + "/M5API/getuid.php");  //サーバーのアドレス
  int httpCode = http.GET();                            //サーバーに接続
  // HTTP応答の処理
  if (httpCode > 0) {                                      //内容があるか確認
    String payload = http.getString();                     //変数にGETの内容代入
    Serial.println("HTTP GET successful");                 //シリアルで通信成功を通知(開発用)
    File file = SPIFFS.open("/uuid.txt", "w");             // SPIFFSに書き込み
    if (file) {                                            //ファイルチェック
      file.print(payload);                                 //ファイルに書き込み
      file.close();                                        //ファイルを閉じる
      Serial.println("Data written to SPIFFS");            //シリアルで書き込み成功を通知(開発用)
      ESP.restart();                                       //再起動
    } else {                                               //if文「または」
      Serial.println("Failed to open file for writing");   //失敗をシリアルで通知
    }                                                      //if文終了
  } else {                                                 //if文「または」
    Serial.print("HTTP GET failed with error code: ");     //シリアルでエラーを通知(開発用)
    Serial.println("HTTP GET failed");                     //シリアルでエラーを通知(開発用)
    Serial.println(http.errorToString(httpCode).c_str());  //シリアルでエラーを通知(開発用)
  }                                                        //if文終了
  http.end();                                              //httpを終了する
}  //void getUUID終了




//RTCに書き込み
//RTCに書き込み
void setNTP2RTC() {
  // NTPサーバーの設定（JST = UTC+9時間、サマータイム=0）
  configTime(JST, 0, "ntp.nict.jp", "ntp.jst.mfeed.ad.jp");
  
  Serial.println("NTP同期中...");
  
  // NTPから時刻取得を待つ（タイムアウト付き）
  int retry_count = 0;
  const int max_retry = 50;  // 最大5秒待つ
  
  while (!getLocalTime(&timeinfo) && retry_count < max_retry) {
    delay(100);  // 100ms待つ
    retry_count++;
    Serial.print(".");
  }
  
  if (retry_count >= max_retry) {
    Serial.println("\nNTP同期に失敗しました");
    return;
  }
  
  Serial.println("\nNTP同期完了");
  
  // デバッグ用：取得した時刻を表示
  Serial.printf("取得した時刻: %04d/%02d/%02d %02d:%02d:%02d (曜日:%s)\n",
                timeinfo.tm_year + 1900,
                timeinfo.tm_mon + 1,
                timeinfo.tm_mday,
                timeinfo.tm_hour,
                timeinfo.tm_min,
                timeinfo.tm_sec,
                week[timeinfo.tm_wday]);
  
  // RTCに設定
  RTCDateTime.date.year = timeinfo.tm_year + 1900;
  RTCDateTime.date.month = timeinfo.tm_mon + 1;
  RTCDateTime.date.date = timeinfo.tm_mday;
  RTCDateTime.date.weekDay = timeinfo.tm_wday;
  RTCDateTime.time.hours = timeinfo.tm_hour;
  RTCDateTime.time.minutes = timeinfo.tm_min;
  RTCDateTime.time.seconds = timeinfo.tm_sec;
  
  M5.Rtc.setDateTime(&RTCDateTime);
  Serial.println("RTCを設定しました");
  
  // 設定確認用：RTCから読み取った時刻を表示
  M5.Rtc.getDateTime(&RTCDateTime);
  Serial.printf("RTC設定確認: %04d/%02d/%02d %02d:%02d:%02d (曜日:%s)\n",
                RTCDateTime.date.year,
                RTCDateTime.date.month,
                RTCDateTime.date.date,
                RTCDateTime.time.hours,
                RTCDateTime.time.minutes,
                RTCDateTime.time.seconds,
                week[RTCDateTime.date.weekDay]);
}

//UUIDを読み込みます
void readUUID() {
  File file = SPIFFS.open("/uuid.txt", "r");            //uuid.txtを開く
  if (!file) {                                          //ファイルチェック
    Serial.println("uuid.txt ファイルが開けません！");  //シリアルで通知(開発用)
    return;                                             //リターン
  }                                                     //if文終了
  uuid = file.readString();                             // ファイルの中身を読み取り、変数に代入
  file.close();                                         //ファイルを閉じる
  Serial.println("読み取ったUUID: " + uuid);            // 読み取ったUUIDをシリアルモニタに表示
}  //void readUUID終了













////////////////////////////////////////////////////////////////////////////////
// オンラインアップデート機能
////////////////////////////////////////////////////////////////////////////////
void getCert() {
  SPIFFS.remove("/root_ca.pem");                                     //root_ca.pemを削除
  HTTPClient http;                                                   //httpクライアントを定義
  http.begin("https://" + host + "/M5API/getCert.php?url=" + host);  //サーバーのアドレス
  int httpCode = http.GET();                                         //サーバーに接続
  if (httpCode > 0) {                                                //内容があるか確認
    String payload = http.getString();                               //変数にGETの内容代入
    Serial.println("HTTP GET successful");                           //シリアルで通信成功を通知(開発用)
    File file = SPIFFS.open("/root_ca.pem", "w");                    // SPIFFSに書き込み
    if (file) {                                                      //ファイルチェック
      file.print(payload);                                           //ファイルに書き込み
      file.close();                                                  //ファイルを閉じる
      Serial.println("Data written to SPIFFS");                      //シリアルで書き込み成功を通知(開発用)
      Serial.println(payload);                                       //シリアルで書き込み成功を通知(開発用)
    } else {                                                         //if文「または」
      Serial.println("Failed to open file for writing");             //失敗をシリアルで通知
    }                                                                //if文終了
  } else {                                                           //if文「または」
    Serial.print("HTTP GET failed with error code: ");               //シリアルでエラーを通知(開発用)
    Serial.println("HTTP GET failed");                               //シリアルでエラーを通知(開発用)
    Serial.println(http.errorToString(httpCode).c_str());            //シリアルでエラーを通知(開発用)
  }                                                                  //if文終了
  http.end();                                                        //httpを終了する
}


void ota() {
  if (updatedNeeded) {
    esp32FOTA.execOTA();
  }
  delay(2000);
}









////////////////////////////////////////////////////////////////////////////////
//jobs
////////////////////////////////////////////////////////////////////////////////
void getSensor() {  //センサーの値を取得する
  // SHT40から温度・湿度を取得
  if (env_sensor_ver == 4) {
    // SHT40の温度・湿度読み取り
    Wire.beginTransmission(0x44);
    Wire.write(0xFD);                   // 高精度測定コマンド
    if (Wire.endTransmission() == 0) {  // 送信成功を確認
      delay(10);

      Wire.requestFrom(0x44, 6);
      if (Wire.available() >= 6) {
        uint16_t temp_raw = (Wire.read() << 8) | Wire.read();
        Wire.read();  // temp CRC (読み飛ばし)
        uint16_t hum_raw = (Wire.read() << 8) | Wire.read();
        Wire.read();  // hum CRC (読み飛ばし)

        // SHT40の正しい変換式
        tmp = -45.0 + 175.0 * temp_raw / 65535.0;
        hum = -6.0 + 125.0 * hum_raw / 65535.0;

        // 範囲チェック
        if (hum < 0) hum = 0;
        if (hum > 100) hum = 100;
      }
    }
  }

  // BMP280から気圧を取得
  Wire.beginTransmission(0x76);
  Wire.write(0xF7);                   // 気圧データレジスタ
  if (Wire.endTransmission() == 0) {  // 送信成功を確認
    Wire.requestFrom(0x76, 3);
    if (Wire.available() >= 3) {
      uint32_t pressure_raw = ((uint32_t)Wire.read() << 12) | ((uint32_t)Wire.read() << 4) | (Wire.read() >> 4);

      // BMP280の場合、キャリブレーションが必要
      // 簡易的な変換（実際はキャリブレーション値を使用）
      pressure = pressure_raw / 256.0;  // 仮の変換式

      // 妥当な範囲チェック（300-1100hPa）
      if (pressure < 300 || pressure > 1100) {
        pressure = 1013.25;  // デフォルト値
      }
    }
  }

  // バッテリー情報取得
  BTVoltage = M5.Power.getBatteryVoltage() / 1000.0;  // mVからVに変換
  BTLevel = M5.Power.getBatteryLevel();               // バッテリー残量を取得

  // 実電圧計算（校正係数適用）
  SoilSensorVoltage = alpha * analogRead(CH1_PIN) * MAX_VOLTAGE / ANALOG_MAX;
}  //void getSensor終了



void uploadData() {
  if (SukenWiFi.isConnected()) {
    HTTPClient http;
    // サーバーにPOSTするJSONデータを作成
    JsonDocument doc;
    doc["BordModel"] = model;
    doc["SystemVersion"] = Version;
    doc["uuid"] = uuid;
    doc["tmp"] = tmp;
    doc["hum"] = hum;
    doc["pressure"] = pressure;
    doc["SoilSensor"] = SoilSensorVoltage;
    doc["BTLevel"] = BTLevel;
    doc["BTVoltage"] = BTVoltage;
    String jsonPayload;
    serializeJson(doc, jsonPayload);
    http.begin("https://" + host + "/M5API/postData.php");
    http.addHeader("Content-Type", "application/json");
    int httpResponseCode = http.POST(jsonPayload);
    if (httpResponseCode > 0) {
      Serial.print("HTTP Response code: ");
      Serial.println(httpResponseCode);
      String response = http.getString();
      Serial.println(response);
    } else {
      Serial.print("HTTP Request failed. Error code: ");
      Serial.println(httpResponseCode);
    }
    http.end();
  }
}



void getSetting() {
  HTTPClient http;
  http.begin("https://" + host + "/M5API/getSetting.php?uuid=" + uuid);

  int httpCode = http.GET();
  if (httpCode > 0) {
    // 成功したらJSONデータを取得する
    String payload = http.getString();
    Serial.println(payload);

    // JSONデータをパースして変数に代入する
    // 以下はArduinoJsonライブラリを使用した例
    JsonDocument doc;
    deserializeJson(doc, payload);

    // DisplayImgUrlの取得
    DisplayImgUrl = doc["DisplayImgUrl"].as<String>();
    Serial.println("DisplayImgUrl: " + DisplayImgUrl);

    // LCDbrightnessの取得
    LCDbrightness = doc["LCDbrightness"].as<int>();
    Serial.println("LCDbrightness: " + String(LCDbrightness));

    // UpdateSpeedの取得
    UpdateSpeed = doc["UpdateSpeed"].as<int>();
    Serial.println("UpdateSpeed: " + String(UpdateSpeed));

    // LEDの取得
    LED = doc["LED"].as<int>();
    Serial.println("LED: " + String(LED));

    // LEDの取得
    BTprotection = doc["BTprotection"].as<int>();
    Serial.println("BTprotection: " + String(BTprotection));

    SystemBackColor = doc["SystemBackColor"].as<int>();
    Serial.println("SystemBackColor: " + String(SystemBackColor));

    SystemTextColor = doc["SystemTextColor"].as<int>();
    Serial.println("SystemTextColor: " + String(SystemTextColor));

    ClockBackColor = doc["ClockBackColor"].as<String>();
    Serial.println("ClockBackColor: " + ClockBackColor);

    ClockTextColor = doc["ClockTextColor"].as<String>();
    Serial.println("ClockTextColor: " + ClockTextColor);

  } else {
    Serial.println("Failed to connect to API");
  }

  http.end();
  // lcd.setBrightness(LCDbrightness);  // 画面の明るさを設定（LVGLに統一のためコメントアウト）
}  //void getSetting終了




void SaveSD() {
  M5.Rtc.getDateTime(&RTCDateTime);  // 現在日時を取得
  String Date = String(RTCDateTime.date.year) + "/" + String(RTCDateTime.date.month) + "/" + String(RTCDateTime.date.date);
  String Time = String(RTCDateTime.time.hours) + ":" + String(RTCDateTime.time.minutes) + ":" + String(RTCDateTime.time.seconds);  // 現在時刻を表示
  JsonDocument doc;
  doc["BordModel"] = model;
  doc["SystemVersion"] = Version;
  doc["tmp"] = tmp;
  doc["hum"] = hum;
  doc["pressure"] = pressure;
  doc["SoilSensor"] = SoilSensorVoltage;
  doc["BTLevel"] = BTLevel;
  doc["BTVoltage"] = BTVoltage;
  doc["Date"] = Date;
  doc["Time"] = Time;

  // JSONを文字列に変換
  String output;
  serializeJson(doc, output);

  // SDカードにデータを書き込む
  File dataFile = SD.open("/data.txt", FILE_APPEND);
  if (dataFile) {
    dataFile.println(output);
    dataFile.close();
    Serial.println("SDカードにセーブしました");
  } else {
    Serial.println("Failed to open data.txt");
  }
}



void UpdateCheck() {
  updatedNeeded = esp32FOTA.execHTTPcheck();
}