import sys
import re
import matplotlib.pyplot as plt
import squarify

def parse_map_file(file_path):
    """
    ESP32 Arduino .map ファイルからグローバル変数を抽出
    サイズが 0 の変数や *fill* は無視
    """
    variables = []

    with open(file_path, "r") as f:
        lines = f.readlines()

    current_name = None
    for line in lines:
        line = line.strip()
        # *fill* は無視
        if line.startswith("*fill*"):
            current_name = None
            continue

        # 変数名行
        if line.startswith(".bss.") or line.startswith(".data.") or line.startswith(".rodata."):
            current_name = line
        else:
            # アドレス + サイズ + オブジェクトファイル
            m = re.match(r"0x[0-9a-fA-F]+\s+0x([0-9a-fA-F]+)\s+.*", line)
            if m and current_name:
                size = int(m.group(1), 16)
                if size > 0:
                    variables.append({"name": current_name, "size": size})
                current_name = None

    return variables

def visualize_variables(variables, top_n=100, output_file="global_vars.png"):
    """
    上位 N 個のグローバル変数をツリーマップで可視化して PNG 出力
    """
    if not variables:
        print("No variables to visualize.")
        return

    # サイズ順にソートして上位 N 個だけ
    variables = sorted(variables, key=lambda v: v["size"], reverse=True)[:top_n]

    sizes = [v["size"] for v in variables]
    labels = [f"{v['name']}\n({v['size']}B)" for v in variables]

    plt.figure(figsize=(16, 10))
    squarify.plot(sizes=sizes, label=labels, alpha=0.8, edgecolor="white")
    plt.axis("off")
    plt.title(f"Top {top_n} ESP32 Global Variables by Size")
    plt.savefig(output_file, dpi=200)
    print(f"Saved visualization as {output_file}")

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python3 visualize_map.py <map_file>")
        sys.exit(1)

    map_file = sys.argv[1]
    vars_list = parse_map_file(map_file)
    print(f"Found {len(vars_list)} global variables in total.")
    visualize_variables(vars_list, top_n=100)

