////////////////////////////////////////////////////////////////////////////////
//ライブラリをインクルードします
////////////////////////////////////////////////////////////////////////////////
// システム定数
#define JST 3600 * 9
#define SD_CS_PIN 4
#define CH1_PIN 35
#define CH2_PIN 36
#define MAX_VOLTAGE 3.3
#define ANALOG_MAX 4095
#include <M5Unified.h>
#include <SD.h>
#include <Wire.h>
#include "Adafruit_Sensor.h"
#include <esp32FOTA.hpp>
#include <TaskScheduler.h>
#include <WiFi.h>
#include <lvgl.h>
//#include "image.h"  //画像ファイルです
#include "html/htmls.h"
#include "state.h"
#include "functions.h"
#include "SukenESPWiFi.h"
#include "pig.h"


// ページ管理
int page_transition = 1;  //ページの遷移
int resetbutton = 0;      //リセットボタンの変数


// グローバル変数
int a;  //Core0のループ検知防止用
TaskHandle_t thp[16];

// システム設定
String model = "MierunCore2";
int Version = 4;
String host = "ecohero.reach-world.net";
String uuid = "0";

// センサーデータ
float tmp = 0;                //温度
float hum = 0;                //湿度
float pressure = 0;           //気圧
float BTVoltage = 0;          //バッテリー電圧
float BTLevel = 0;            //バッテリー残量
float SoilSensorVoltage = 0;  // CH1の真の電圧


// ユーザー設定
int env_sensor_ver = 1;
float alpha = 1.0;  // 倍率
String DisplayImgUrl;
int LCDbrightness = 100;
int LED = 1;
int BTprotection = 1;
int UpdateSpeed = 1500;  //ミリ秒

// 表示設定（フォント固定）
int SystemBackColor = BLACK;
int SystemTextColor = WHITE;
String ClockBackColor = "BLACK";
String ClockTextColor = "CYAN";




bool updatedNeeded = false;


esp32FOTA esp32FOTA(model, Version, false, true);
String manifest_url_string = "https://" + host + "/M5API/OTA.php";
const char *manifest_url = manifest_url_string.c_str();
////////////////////////////////////////////////////////////////////////////////
//RTC関係の部分
////////////////////////////////////////////////////////////////////////////////
const char *week[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };  // 曜日を定義
m5::rtc_datetime_t RTCDateTime;                                             //RTC関連の設定(M5Unified)
struct tm timeinfo;                                                         //RTC関連の設定


const String motd = R"=====(
プロダクト名:みえるん
FabQuest2023-Gropu4-エコヒーローズ

メンバー
・佐々木理一
・田辺奈々葉
・吉仲遼
・山岸永実
・西巻美結
・古川健

スペシャルサンクス
・古川嘉徳
・渡辺ゆうか
・山本修
・サポーターの皆様
・松本信夫
・長谷川孝一
・平野さん
・川原さん

協力しれくださった皆様本当にありがとうございます。
)=====";
// サーバー関連
WebServer server(80);


////////////////////////////////////////////////////////////////////////////////
//LVGL関係の部分
////////////////////////////////////////////////////////////////////////////////

#define TFT_HOR_RES 320
#define TFT_VER_RES 240
#define TFT_ROTATION LV_DISPLAY_ROTATION_0

/*LVGL draw into this buffer, smaller buffer to save memory*/
#define DRAW_BUF_SIZE (TFT_HOR_RES * 3 * (LV_COLOR_DEPTH / 8))  // 60行分のバッファ

uint32_t draw_buf[DRAW_BUF_SIZE / 4];


// 動的画面管理用変数
lv_obj_t *current_screen_obj = nullptr;
lv_obj_t *current_canvas = nullptr;
lv_obj_t *temp_value_label = nullptr;
lv_obj_t *humi_value_label = nullptr;
lv_obj_t *soil_value_label = nullptr;
lv_obj_t *pressure_label = nullptr;
lv_obj_t *battery_label = nullptr;
lv_obj_t *date_canvas = nullptr;
lv_obj_t *seconds_canvas = nullptr;
lv_obj_t *weekday_canvas = nullptr;

// グローバル変数でデータラベルを管理
static lv_obj_t *sys_data_labels[10] = {nullptr};


// 共有文字列バッファ（PROGMEMに配置）
const char PROGMEM NAV_TEXT[] = "Button A: Previous | Button B: Next";
const char PROGMEM CLOCK_FORMAT[] = "%04d-%02d-%02d(%s) %02d:%02d:%02d";
const char PROGMEM SENSOR_FORMAT[] = "Temp: %2.1f\nHumi: %2.0f%%\nPressure: %2.0fhPa\nBattery: %2.2fV\nBattery: %3.0f%%\nSoilSensor: %1.2fV";
const char PROGMEM SYSTEM_FORMAT[] = "RAM Usage: %.2f%%\nRAM FreeMemory: %.2f MB\nCPU Usage: %.2f%%";
const char PROGMEM SYSTEM_DATA_FORMAT[] = "Model: %s\nSystemVer: %s\nUUID: %s\nEnvSensor-Ver: %s\nLCDbrightness: %d\nUpdateSpeed: %d\nIP: %s\nWiFi: %s\nMAC: %s";

// 共有文字列バッファ
char shared_text_buffer[256];


// Current screen index
int current_screen = 0;

/* LVGL calls it when a rendered image needs to copied to the display*/
void my_disp_flush(lv_display_t *disp, const lv_area_t *area, uint8_t *px_map) {
  uint32_t width = (area->x2 - area->x1 + 1);
  uint32_t height = (area->y2 - area->y1 + 1);

  lv_draw_sw_rgb565_swap(px_map, width * height);
  M5.Display.pushImageDMA<uint16_t>(area->x1, area->y1, width, height, (uint16_t *)px_map);
  lv_display_flush_ready(disp);
}

/*Read the touchpad*/
void my_touchpad_read(lv_indev_t *indev, lv_indev_data_t *data) {
  lgfx::touch_point_t tp;
  M5.update();
  if (M5.Display.getTouch(&tp)) {
    data->state = LV_INDEV_STATE_PRESSED;
    data->point.x = tp.x;
    data->point.y = tp.y;
  } else {
    data->state = LV_INDEV_STATE_RELEASED;
  }
}

/*use Arduinos millis() as tick source*/
static uint32_t my_tick(void) {
  return millis();
}


////////////////////////////////////////////////////////////////////////////////
//Screen関係の部分（動的生成）
////////////////////////////////////////////////////////////////////////////////

// 画面1を作成
void create_screen1() {
  // 既存の画面を破棄
  destroy_current_screen();

  // 新しい画面オブジェクトを作成
  current_screen_obj = lv_obj_create(NULL);

  // スタイル設定を明示的に適用
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // 画像オブジェクトを作成（pig.c の画像を表示）
  lv_obj_t *eye_img = lv_img_create(current_screen_obj);
  lv_img_set_src(eye_img, &pig);
  lv_obj_align(eye_img, LV_ALIGN_CENTER, 0, -10);
  Serial.println("Screen 1 created successfully");
}


// Create screen 2 - Enhanced Clock Display
void create_screen2() {
    destroy_current_screen();
    current_screen_obj = lv_obj_create(NULL);
    
    // グラデーション背景の設定
    lv_obj_remove_style_all(current_screen_obj);
    lv_obj_set_style_bg_color(current_screen_obj, lv_color_hex(0x0a0a2e), 0);
    lv_obj_set_style_bg_grad_color(current_screen_obj, lv_color_hex(0x16213e), 0);
    lv_obj_set_style_bg_grad_dir(current_screen_obj, LV_GRAD_DIR_VER, 0);
    lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);
    
    // メインコンテナ
    lv_obj_t *main_container = lv_obj_create(current_screen_obj);
    lv_obj_set_size(main_container, 300, 200);
    lv_obj_align(main_container, LV_ALIGN_CENTER, 0, 0);
    lv_obj_set_style_bg_color(main_container, lv_color_hex(0x1e1e3f), 0);
    lv_obj_set_style_bg_opa(main_container, LV_OPA_80, 0);
    lv_obj_set_style_radius(main_container, 15, 0);
    lv_obj_set_style_border_color(main_container, lv_color_hex(0x4169e1), 0);
    lv_obj_set_style_border_width(main_container, 2, 0);
    lv_obj_set_style_border_opa(main_container, LV_OPA_50, 0);
    lv_obj_set_style_shadow_width(main_container, 20, 0);
    lv_obj_set_style_shadow_color(main_container, lv_color_black(), 0);
    lv_obj_set_style_shadow_opa(main_container, LV_OPA_50, 0);
    lv_obj_clear_flag(main_container, LV_OBJ_FLAG_SCROLLABLE);
    
    // 時刻表示ラベル（大きなフォント）
    lv_obj_t *time_label = lv_label_create(main_container);
    lv_obj_set_size(time_label, 260, LV_SIZE_CONTENT);
    lv_obj_align(time_label, LV_ALIGN_TOP_MID, 0, 20);
    lv_obj_set_style_text_font(time_label, &lv_font_montserrat_28, 0);
    lv_obj_set_style_text_color(time_label, lv_color_hex(0x00ff9f), 0);
    lv_obj_set_style_text_align(time_label, LV_TEXT_ALIGN_CENTER, 0);
    
    // 日付表示ラベル
    lv_obj_t *date_label = lv_label_create(main_container);
    lv_obj_set_size(date_label, 260, LV_SIZE_CONTENT);
    lv_obj_align(date_label, LV_ALIGN_TOP_MID, 0, 65);
    lv_obj_set_style_text_font(date_label, &lv_font_montserrat_16, 0);
    lv_obj_set_style_text_color(date_label, lv_color_hex(0x87ceeb), 0);
    lv_obj_set_style_text_align(date_label, LV_TEXT_ALIGN_CENTER, 0);
    
    // 曜日表示ラベル
    lv_obj_t *weekday_label = lv_label_create(main_container);
    lv_obj_set_size(weekday_label, 260, LV_SIZE_CONTENT);
    lv_obj_align(weekday_label, LV_ALIGN_TOP_MID, 0, 90);
    lv_obj_set_style_text_font(weekday_label, &lv_font_montserrat_14, 0);
    lv_obj_set_style_text_color(weekday_label, lv_color_hex(0xffd700), 0);
    lv_obj_set_style_text_align(weekday_label, LV_TEXT_ALIGN_CENTER, 0);
    
    // デジタル風装飾ライン
    lv_obj_t *line1 = lv_obj_create(main_container);
    lv_obj_set_size(line1, 200, 2);
    lv_obj_align(line1, LV_ALIGN_TOP_MID, 0, 55);
    lv_obj_set_style_bg_color(line1, lv_color_hex(0x4169e1), 0);
    lv_obj_set_style_bg_opa(line1, LV_OPA_70, 0);
    lv_obj_set_style_radius(line1, 1, 0);
    
    lv_obj_t *line2 = lv_obj_create(main_container);
    lv_obj_set_size(line2, 150, 1);
    lv_obj_align(line2, LV_ALIGN_TOP_MID, 0, 115);
    lv_obj_set_style_bg_color(line2, lv_color_hex(0x4169e1), 0);
    lv_obj_set_style_bg_opa(line2, LV_OPA_50, 0);
    lv_obj_set_style_radius(line2, 1, 0);
    
    // 角のアクセント
    for(int i = 0; i < 4; i++) {
        lv_obj_t *accent = lv_obj_create(main_container);
        lv_obj_set_size(accent, 15, 15);
        lv_obj_set_style_bg_color(accent, lv_color_hex(0x00ff9f), 0);
        lv_obj_set_style_bg_opa(accent, LV_OPA_60, 0);
        lv_obj_set_style_radius(accent, 7, 0);
        
        switch(i) {
            case 0: lv_obj_align(accent, LV_ALIGN_TOP_LEFT, 10, 10); break;
            case 1: lv_obj_align(accent, LV_ALIGN_TOP_RIGHT, -10, 10); break;
            case 2: lv_obj_align(accent, LV_ALIGN_BOTTOM_LEFT, 10, -10); break;
            case 3: lv_obj_align(accent, LV_ALIGN_BOTTOM_RIGHT, -10, -10); break;
        }
    }
    
    // グローバル参照用に保存（複数のラベルを管理するための構造体が必要）
    // ここでは時刻ラベルをメインとして保存
    current_canvas = time_label;
    
    // 追加のラベル参照用（グローバル変数として定義が必要）
    date_canvas = date_label;
    weekday_canvas = weekday_label;
    
    // 初期表示
    update_clock_display();
    Serial.println("Enhanced Screen 2 created successfully");
}

// Enhanced clock display update
void update_clock_display() {
    if (current_canvas == nullptr) return;
    
    // RTCから現在時刻を取得
    M5.Rtc.getDateTime(&RTCDateTime);
    
    // 時刻表示（HH:MM:SS）
    char time_str[20];
    snprintf(time_str, sizeof(time_str), "%02d:%02d:%02d", 
             RTCDateTime.time.hours, RTCDateTime.time.minutes, RTCDateTime.time.seconds);
    lv_label_set_text((lv_obj_t *)current_canvas, time_str);
    
    // 日付表示（YYYY/MM/DD）
    if (date_canvas != nullptr) {
        char date_str[20];
        snprintf(date_str, sizeof(date_str), "%04d/%02d/%02d", 
                 RTCDateTime.date.year, RTCDateTime.date.month, RTCDateTime.date.date);
        lv_label_set_text((lv_obj_t *)date_canvas, date_str);
    }
    
    // 曜日表示
    if (weekday_canvas != nullptr) {
        const char* weekdays[] = {"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"};
        lv_label_set_text((lv_obj_t *)weekday_canvas, weekdays[RTCDateTime.date.weekDay]);
    }
}

// Alternative simpler design version
void create_screen2_simple() {
    destroy_current_screen();
    current_screen_obj = lv_obj_create(NULL);
    
    // ダークテーマ背景
    lv_obj_remove_style_all(current_screen_obj);
    lv_obj_set_style_bg_color(current_screen_obj, lv_color_hex(0x1a1a1a), 0);
    lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);
    
    // 時刻表示（大きく中央に）
    lv_obj_t *clock_label = lv_label_create(current_screen_obj);
    lv_obj_set_size(clock_label, 320, LV_SIZE_CONTENT);
    lv_obj_align(clock_label, LV_ALIGN_CENTER, 0, -30);
    lv_obj_set_style_text_font(clock_label, &lv_font_montserrat_32, 0);
    lv_obj_set_style_text_color(clock_label, lv_color_white(), 0);
    lv_obj_set_style_text_align(clock_label, LV_TEXT_ALIGN_CENTER, 0);
    
    // 日付表示（時刻の下）
    lv_obj_t *date_label = lv_label_create(current_screen_obj);
    lv_obj_set_size(date_label, 320, LV_SIZE_CONTENT);
    lv_obj_align(date_label, LV_ALIGN_CENTER, 0, 20);
    lv_obj_set_style_text_font(date_label, &lv_font_montserrat_18, 0);
    lv_obj_set_style_text_color(date_label, lv_color_hex(0xaaaaaa), 0);
    lv_obj_set_style_text_align(date_label, LV_TEXT_ALIGN_CENTER, 0);
    
    // アクセントライン
    lv_obj_t *accent_line = lv_obj_create(current_screen_obj);
    lv_obj_set_size(accent_line, 100, 3);
    lv_obj_align(accent_line, LV_ALIGN_CENTER, 0, 45);
    lv_obj_set_style_bg_color(accent_line, lv_color_hex(0x00aaff), 0);
    lv_obj_set_style_bg_opa(accent_line, LV_OPA_COVER, 0);
    lv_obj_set_style_radius(accent_line, 2, 0);
    
    current_canvas = clock_label;
    date_canvas = date_label;
    
    update_clock_display_simple();
    Serial.println("Simple Screen 2 created successfully");
}

// Simple update function
void update_clock_display_simple() {
    if (current_canvas == nullptr) return;
    
    M5.Rtc.getDateTime(&RTCDateTime);
    
    // 時刻表示
    char time_str[10];
    snprintf(time_str, sizeof(time_str), "%02d:%02d:%02d", 
             RTCDateTime.time.hours, RTCDateTime.time.minutes, RTCDateTime.time.seconds);
    lv_label_set_text((lv_obj_t *)current_canvas, time_str);
    
    // 日付表示
    if (date_canvas != nullptr) {
        char date_str[30];
        const char* weekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
        snprintf(date_str, sizeof(date_str), "%04d/%02d/%02d (%s)", 
                 RTCDateTime.date.year, RTCDateTime.date.month, RTCDateTime.date.date,
                 weekdays[RTCDateTime.date.weekDay]);
        lv_label_set_text((lv_obj_t *)date_canvas, date_str);
    }
}
void create_screen3() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_black(), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // メインコンテナ（320x240の画面全体を使用）
  lv_obj_t *main_container = lv_obj_create(current_screen_obj);
  lv_obj_set_size(main_container, 320, 240); // 画面全体を使用
  lv_obj_align(main_container, LV_ALIGN_TOP_LEFT, 0, 0);
  lv_obj_set_style_bg_opa(main_container, LV_OPA_TRANSP, 0);
  lv_obj_set_style_border_width(main_container, 0, 0);
  lv_obj_set_style_pad_all(main_container, 0, 0); // 余白を0に

  // 上段：温度とバッテリー（横並び）
  create_temperature_display_m5(main_container);
  create_battery_display_m5(main_container);
  
  // 中段：湿度と気圧（横並び）
  create_humidity_display_m5(main_container);
  create_pressure_display_m5(main_container);
  
  // 下段：土壌水分センサー（全幅）
  create_soil_moisture_display_m5(main_container);

  // 初期表示
  update_sensor_display();

  Serial.println("Screen 3 (M5Core2 Optimized) created successfully");
}

// 温度表示を作成（M5Core2用）
void create_temperature_display_m5(lv_obj_t *parent) {
  // 温度表示コンテナ（上段左）
  lv_obj_t *temp_container = lv_obj_create(parent);
  lv_obj_set_size(temp_container, 158, 75);
  lv_obj_align(temp_container, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_bg_color(temp_container, lv_color_hex(0x1a1a1a), 0);
  lv_obj_set_style_border_width(temp_container, 1, 0);
  lv_obj_set_style_border_color(temp_container, lv_color_hex(0x444444), 0);
  lv_obj_set_style_radius(temp_container, 8, 0);
  lv_obj_set_style_pad_all(temp_container, 5, 0);
  
  // 温度アイコン・タイトル
  lv_obj_t *temp_title = lv_label_create(temp_container);
  lv_label_set_text(temp_title, LV_SYMBOL_CHARGE " Temperature");
  lv_obj_align(temp_title, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_text_color(temp_title, lv_color_hex(0xFF6666), 0);
  lv_obj_set_style_text_font(temp_title, &lv_font_montserrat_12, 0);
  
  // 温度値表示
  temp_value_label = lv_label_create(temp_container);
  lv_label_set_text(temp_value_label, "25.0°C");
  lv_obj_align(temp_value_label, LV_ALIGN_CENTER, 0, 5);
  lv_obj_set_style_text_color(temp_value_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(temp_value_label, &lv_font_montserrat_20, 0);
}

// バッテリー表示を作成（M5Core2用）
void create_battery_display_m5(lv_obj_t *parent) {
  lv_obj_t *batt_container = lv_obj_create(parent);
  lv_obj_set_size(batt_container, 158, 75);
  lv_obj_align(batt_container, LV_ALIGN_TOP_RIGHT, -2, 2);
  lv_obj_set_style_bg_color(batt_container, lv_color_hex(0x1a1a1a), 0);
  lv_obj_set_style_border_width(batt_container, 1, 0);
  lv_obj_set_style_border_color(batt_container, lv_color_hex(0x444444), 0);
  lv_obj_set_style_radius(batt_container, 8, 0);
  lv_obj_set_style_pad_all(batt_container, 5, 0);
  
  lv_obj_t *batt_title = lv_label_create(batt_container);
  lv_label_set_text(batt_title, LV_SYMBOL_BATTERY_FULL " Battery");
  lv_obj_align(batt_title, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_text_color(batt_title, lv_color_hex(0x66FF66), 0);
  lv_obj_set_style_text_font(batt_title, &lv_font_montserrat_12, 0);
  
  battery_label = lv_label_create(batt_container);
  lv_label_set_text(battery_label, "3.7V\n85%");
  lv_obj_align(battery_label, LV_ALIGN_CENTER, 0, 5);
  lv_obj_set_style_text_color(battery_label, lv_color_hex(0x33FF33), 0);
  lv_obj_set_style_text_font(battery_label, &lv_font_montserrat_14, 0);
  lv_obj_set_style_text_align(battery_label, LV_TEXT_ALIGN_CENTER, 0);
}

// 湿度表示を作成（M5Core2用）
void create_humidity_display_m5(lv_obj_t *parent) {
  lv_obj_t *humi_container = lv_obj_create(parent);
  lv_obj_set_size(humi_container, 158, 75);
  lv_obj_align(humi_container, LV_ALIGN_TOP_LEFT, 2, 80);
  lv_obj_set_style_bg_color(humi_container, lv_color_hex(0x1a1a1a), 0);
  lv_obj_set_style_border_width(humi_container, 1, 0);
  lv_obj_set_style_border_color(humi_container, lv_color_hex(0x444444), 0);
  lv_obj_set_style_radius(humi_container, 8, 0);
  lv_obj_set_style_pad_all(humi_container, 5, 0);
  
  lv_obj_t *humi_title = lv_label_create(humi_container);
  lv_label_set_text(humi_title, LV_SYMBOL_TINT " Humidity");
  lv_obj_align(humi_title, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_text_color(humi_title, lv_color_hex(0x6699FF), 0);
  lv_obj_set_style_text_font(humi_title, &lv_font_montserrat_12, 0);
  
  humi_value_label = lv_label_create(humi_container);
  lv_label_set_text(humi_value_label, "60%");
  lv_obj_align(humi_value_label, LV_ALIGN_CENTER, 0, 5);
  lv_obj_set_style_text_color(humi_value_label, lv_color_hex(0x3399FF), 0);
  lv_obj_set_style_text_font(humi_value_label, &lv_font_montserrat_20, 0);
}

// 気圧表示を作成（M5Core2用）
void create_pressure_display_m5(lv_obj_t *parent) {
  lv_obj_t *press_container = lv_obj_create(parent);
  lv_obj_set_size(press_container, 158, 75);
  lv_obj_align(press_container, LV_ALIGN_TOP_RIGHT, -2, 80);
  lv_obj_set_style_bg_color(press_container, lv_color_hex(0x1a1a1a), 0);
  lv_obj_set_style_border_width(press_container, 1, 0);
  lv_obj_set_style_border_color(press_container, lv_color_hex(0x444444), 0);
  lv_obj_set_style_radius(press_container, 8, 0);
  lv_obj_set_style_pad_all(press_container, 5, 0);
  
  lv_obj_t *press_title = lv_label_create(press_container);
  lv_label_set_text(press_title, LV_SYMBOL_GPS " Pressure");
  lv_obj_align(press_title, LV_ALIGN_TOP_LEFT, 2, 2);
  lv_obj_set_style_text_color(press_title, lv_color_hex(0xFFAA33), 0);
  lv_obj_set_style_text_font(press_title, &lv_font_montserrat_12, 0);
  
  pressure_label = lv_label_create(press_container);
  lv_label_set_text(pressure_label, "1013\nhPa");
  lv_obj_align(pressure_label, LV_ALIGN_CENTER, 0, 5);
  lv_obj_set_style_text_color(pressure_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(pressure_label, &lv_font_montserrat_16, 0);
  lv_obj_set_style_text_align(pressure_label, LV_TEXT_ALIGN_CENTER, 0);
}

// 土壌水分表示を作成（M5Core2用・全幅）
void create_soil_moisture_display_m5(lv_obj_t *parent) {
  lv_obj_t *soil_container = lv_obj_create(parent);
  lv_obj_set_size(soil_container, 316, 70);
  lv_obj_align(soil_container, LV_ALIGN_BOTTOM_MID, 0, -2);
  lv_obj_set_style_bg_color(soil_container, lv_color_hex(0x1a1a1a), 0);
  lv_obj_set_style_border_width(soil_container, 1, 0);
  lv_obj_set_style_border_color(soil_container, lv_color_hex(0x444444), 0);
  lv_obj_set_style_radius(soil_container, 8, 0);
  lv_obj_set_style_pad_all(soil_container, 5, 0);
  
  lv_obj_t *soil_title = lv_label_create(soil_container);
  lv_label_set_text(soil_title, LV_SYMBOL_REFRESH " Soil Moisture");
  lv_obj_align(soil_title, LV_ALIGN_TOP_LEFT, 5, 3);
  lv_obj_set_style_text_color(soil_title, lv_color_hex(0x66CCFF), 0);
  lv_obj_set_style_text_font(soil_title, &lv_font_montserrat_12, 0);
  
  // 土壌水分値表示（中央に大きく）
  soil_value_label = lv_label_create(soil_container);
  lv_label_set_text(soil_value_label, "1.20V - Dry");
  lv_obj_align(soil_value_label, LV_ALIGN_CENTER, 0, 5);
  lv_obj_set_style_text_color(soil_value_label, lv_color_white(), 0);
  lv_obj_set_style_text_font(soil_value_label, &lv_font_montserrat_20, 0);
  lv_obj_set_style_text_align(soil_value_label, LV_TEXT_ALIGN_CENTER, 0);
}

// センサー表示更新関数（M5Core2最適化版）
void update_sensor_display() {
  if (temp_value_label == nullptr || soil_value_label == nullptr) return;

  // 温度値の更新
  if (temp_value_label) {
    snprintf(shared_text_buffer, 20, "%.1f°C", tmp);
    lv_label_set_text(temp_value_label, shared_text_buffer);
  }
  
  // 湿度値の更新
  if (humi_value_label) {
    snprintf(shared_text_buffer, 20, "%.0f%%", hum);
    lv_label_set_text(humi_value_label, shared_text_buffer);
  }
  
  // 気圧値の更新
  if (pressure_label) {
    snprintf(shared_text_buffer, 20, "%.0f\nhPa", pressure);
    lv_label_set_text(pressure_label, shared_text_buffer);
  }
  
  // 土壌水分値と状態の更新
  if (soil_value_label) {
    const char* status_text;
    lv_color_t status_color;
    
    if (SoilSensorVoltage > 1.8) {
      status_text = "Very Dry";
      status_color = lv_color_hex(0xFF3333);
    } else if (SoilSensorVoltage > 1.4) {
      status_text = "Dry";
      status_color = lv_color_hex(0xFFAA33);
    } else if (SoilSensorVoltage > 1.0) {
      status_text = "Moist";
      status_color = lv_color_hex(0x33AAFF);
    } else {
      status_text = "Wet";
      status_color = lv_color_hex(0x0066FF);
    }
    
    snprintf(shared_text_buffer, 40, "%.2fV - %s", SoilSensorVoltage, status_text);
    lv_label_set_text(soil_value_label, shared_text_buffer);
    lv_obj_set_style_text_color(soil_value_label, status_color, 0);
  }
  
  // バッテリー値の更新（色も変更）
  if (battery_label) {
    snprintf(shared_text_buffer, 30, "%.2fV\n%.0f%%", BTVoltage, BTLevel);
    lv_label_set_text(battery_label, shared_text_buffer);
    
    // バッテリーレベルに応じて色を変更
    lv_color_t battery_color;
    if (BTLevel > 60) {
      battery_color = lv_color_hex(0x33FF33); // 緑
    } else if (BTLevel > 30) {
      battery_color = lv_color_hex(0xFFAA00); // オレンジ
    } else {
      battery_color = lv_color_hex(0xFF3333); // 赤
    }
    lv_obj_set_style_text_color(battery_label, battery_color, 0);
  }
}

// Create screen 4 - Memory Monitor (M5Core2 320x240 改善版)
void create_screen4() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定（ダークテーマ）
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_hex(0x0d1117), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);

  // タイトルヘッダー
  lv_obj_t *header = lv_obj_create(current_screen_obj);
  lv_obj_set_size(header, 320, 20);
  lv_obj_align(header, LV_ALIGN_TOP_MID, 0, 0);
  lv_obj_set_style_bg_color(header, lv_color_hex(0x161b22), 0);
  lv_obj_set_style_border_width(header, 0, 0);
  lv_obj_set_style_radius(header, 0, 0);
  lv_obj_set_style_pad_all(header, 2, 0);
  lv_obj_set_scrollbar_mode(header, LV_SCROLLBAR_MODE_OFF);

  lv_obj_t *title_label = lv_label_create(header);
  lv_label_set_text(title_label, "Memory Monitor");
  lv_obj_center(title_label);
  lv_obj_set_style_text_color(title_label, lv_color_hex(0xf0f6fc), 0);
  lv_obj_set_style_text_font(title_label, &lv_font_montserrat_12, 0);

  // メインコンテナ（220px = 240 - 20）
  lv_obj_t *main_container = lv_obj_create(current_screen_obj);
  lv_obj_set_size(main_container, 320, 220);
  lv_obj_align(main_container, LV_ALIGN_BOTTOM_MID, 0, 0);
  lv_obj_set_style_bg_color(main_container, lv_color_hex(0x0d1117), 0);
  lv_obj_set_style_border_width(main_container, 0, 0);
  lv_obj_set_style_radius(main_container, 0, 0);
  lv_obj_set_style_pad_all(main_container, 4, 0);
  lv_obj_set_scrollbar_mode(main_container, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(main_container, LV_OBJ_FLAG_SCROLLABLE);

  // RAMメイン表示セクション（上部 - 66px）
  lv_obj_t *ram_section = lv_obj_create(main_container);
  lv_obj_set_size(ram_section, 312, 66);
  lv_obj_align(ram_section, LV_ALIGN_TOP_MID, 0, 0);
  lv_obj_set_style_bg_color(ram_section, lv_color_hex(0x161b22), 0);
  lv_obj_set_style_border_width(ram_section, 1, 0);
  lv_obj_set_style_border_color(ram_section, lv_color_hex(0x30363d), 0);
  lv_obj_set_style_radius(ram_section, 4, 0);
  lv_obj_set_style_pad_all(ram_section, 6, 0);
  lv_obj_set_scrollbar_mode(ram_section, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(ram_section, LV_OBJ_FLAG_SCROLLABLE);

  // RAMタイトルと使用率を横並びで配置
  lv_obj_t *ram_header = lv_obj_create(ram_section);
  lv_obj_set_size(ram_header, 300, 16);
  lv_obj_align(ram_header, LV_ALIGN_TOP_MID, 0, 0);
  lv_obj_set_style_bg_opa(ram_header, LV_OPA_0, 0);
  lv_obj_set_style_border_width(ram_header, 0, 0);
  lv_obj_set_style_pad_all(ram_header, 0, 0);

  lv_obj_t *ram_title = lv_label_create(ram_header);
  lv_label_set_text(ram_title, "Memory Usage");
  lv_obj_align(ram_title, LV_ALIGN_LEFT_MID, 0, 0);
  lv_obj_set_style_text_color(ram_title, lv_color_hex(0x4fc3f7), 0);
  lv_obj_set_style_text_font(ram_title, &lv_font_montserrat_10, 0);

  // RAM使用率パーセント（右側、中央寄せ）
  lv_obj_t *ram_percent = lv_label_create(ram_header);
  lv_label_set_text(ram_percent, "0%");
  lv_obj_align(ram_percent, LV_ALIGN_RIGHT_MID, 0, 0);
  lv_obj_set_style_text_color(ram_percent, lv_color_hex(0x4fc3f7), 0);
  lv_obj_set_style_text_font(ram_percent, &lv_font_montserrat_12, 0);

  // RAM プログレスバー（適切な間隔で配置）
  lv_obj_t *ram_bar = lv_bar_create(ram_section);
  lv_obj_set_size(ram_bar, 300, 12);
  lv_obj_align(ram_bar, LV_ALIGN_TOP_MID, 0, 18);
  lv_obj_set_style_bg_color(ram_bar, lv_color_hex(0x21262d), LV_PART_MAIN);
  lv_obj_set_style_bg_color(ram_bar, lv_color_hex(0x4fc3f7), LV_PART_INDICATOR);
  lv_obj_set_style_radius(ram_bar, 6, 0);
  lv_obj_set_style_border_width(ram_bar, 1, LV_PART_MAIN);
  lv_obj_set_style_border_color(ram_bar, lv_color_hex(0x30363d), LV_PART_MAIN);
  lv_bar_set_range(ram_bar, 0, 100);

  // RAM詳細情報（プログレスバーの下に適切な間隔）
  lv_obj_t *ram_detail = lv_label_create(ram_section);
  lv_obj_set_size(ram_detail, 300, 32);
  lv_obj_align(ram_detail, LV_ALIGN_TOP_MID, 0, 34);
  lv_obj_set_style_text_font(ram_detail, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(ram_detail, lv_color_hex(0xe6edf3), 0);
  lv_obj_set_style_text_align(ram_detail, LV_TEXT_ALIGN_CENTER, 0);
  lv_label_set_long_mode(ram_detail, LV_LABEL_LONG_WRAP);

  // ヒープ情報セクション（中央 - 66px）
  lv_obj_t *heap_section = lv_obj_create(main_container);
  lv_obj_set_size(heap_section, 312, 66);
  lv_obj_set_pos(heap_section, 0, 68);
  lv_obj_set_style_bg_color(heap_section, lv_color_hex(0x161b22), 0);
  lv_obj_set_style_border_width(heap_section, 1, 0);
  lv_obj_set_style_border_color(heap_section, lv_color_hex(0x30363d), 0);
  lv_obj_set_style_radius(heap_section, 4, 0);
  lv_obj_set_style_pad_all(heap_section, 6, 0);
  lv_obj_set_scrollbar_mode(heap_section, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(heap_section, LV_OBJ_FLAG_SCROLLABLE);

  lv_obj_t *heap_title = lv_label_create(heap_section);
  lv_label_set_text(heap_title, "Heap Details");
  lv_obj_align(heap_title, LV_ALIGN_TOP_LEFT, 0, 0);
  lv_obj_set_style_text_color(heap_title, lv_color_hex(0x7c3aed), 0);
  lv_obj_set_style_text_font(heap_title, &lv_font_montserrat_10, 0);

  lv_obj_t *heap_info_label = lv_label_create(heap_section);
  lv_obj_set_size(heap_info_label, 300, 42);
  lv_obj_align(heap_info_label, LV_ALIGN_TOP_MID, 0, 16);
  lv_obj_set_style_text_font(heap_info_label, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(heap_info_label, lv_color_hex(0xe6edf3), 0);
  lv_obj_set_style_text_align(heap_info_label, LV_TEXT_ALIGN_CENTER, 0);
  lv_label_set_long_mode(heap_info_label, LV_LABEL_LONG_WRAP);

  // システム情報セクション（下部 - 62px）
  lv_obj_t *info_section = lv_obj_create(main_container);
  lv_obj_set_size(info_section, 312, 62);
  lv_obj_set_pos(info_section, 0, 136);
  lv_obj_set_style_bg_color(info_section, lv_color_hex(0x161b22), 0);
  lv_obj_set_style_border_width(info_section, 1, 0);
  lv_obj_set_style_border_color(info_section, lv_color_hex(0x30363d), 0);
  lv_obj_set_style_radius(info_section, 4, 0);
  lv_obj_set_style_pad_all(info_section, 6, 0);
  lv_obj_set_scrollbar_mode(info_section, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(info_section, LV_OBJ_FLAG_SCROLLABLE);

  lv_obj_t *info_title = lv_label_create(info_section);
  lv_label_set_text(info_title, "System Info");
  lv_obj_align(info_title, LV_ALIGN_TOP_LEFT, 0, 0);
  lv_obj_set_style_text_color(info_title, lv_color_hex(0xff6b6b), 0);
  lv_obj_set_style_text_font(info_title, &lv_font_montserrat_10, 0);

  lv_obj_t *system_info_label = lv_label_create(info_section);
  lv_obj_set_size(system_info_label, 300, 38);
  lv_obj_align(system_info_label, LV_ALIGN_TOP_MID, 0, 16);
  lv_obj_set_style_text_font(system_info_label, &lv_font_montserrat_10, 0);
  lv_obj_set_style_text_color(system_info_label, lv_color_hex(0xe6edf3), 0);
  lv_obj_set_style_text_align(system_info_label, LV_TEXT_ALIGN_CENTER, 0);
  lv_label_set_long_mode(system_info_label, LV_LABEL_LONG_WRAP);

  // 更新用の参照を保存
  current_canvas = system_info_label;
  
  // ユーザーデータに各コンポーネントの参照を保存
  static lv_obj_t* components[4];
  components[0] = ram_bar;
  components[1] = ram_percent;
  components[2] = ram_detail;
  components[3] = heap_info_label;
  lv_obj_set_user_data(current_screen_obj, components);

  // 初期表示
  update_system_display();

  Serial.println("Screen 4 created - Memory Monitor 320x240 optimized layout");
}

// システム情報表示更新関数（改善版）
void update_system_display() {
  if (current_canvas == nullptr || current_screen_obj == nullptr) return;

  // メモリ情報の詳細取得
  uint32_t freeRAM = ESP.getFreeHeap();
  uint32_t totalRAM = ESP.getHeapSize();
  uint32_t minFreeRAM = ESP.getMinFreeHeap();
  uint32_t maxAllocRAM = ESP.getMaxAllocHeap();
  
  float ramUsage = ((float)(totalRAM - freeRAM) / totalRAM) * 100;
  float freeMemoryMB = (float)freeRAM / 1024 / 1024;
  float totalMemoryMB = (float)totalRAM / 1024 / 1024;
  float usedMemoryMB = totalMemoryMB - freeMemoryMB;

  // コンポーネントの参照を取得
  lv_obj_t** components = (lv_obj_t**)lv_obj_get_user_data(current_screen_obj);
  if (components != nullptr) {
    lv_obj_t *ram_bar = components[0];
    lv_obj_t *ram_percent = components[1];
    lv_obj_t *ram_detail = components[2];
    lv_obj_t *heap_info_label = components[3];

    // RAM使用率の更新
    lv_bar_set_value(ram_bar, (int)ramUsage, LV_ANIM_ON);
    char ramText[10];
    snprintf(ramText, sizeof(ramText), "%.1f%%", ramUsage);
    lv_label_set_text(ram_percent, ramText);

    // RAM使用率に応じて色を変更（プログレスバーとパーセンテージの色を同期）
    lv_color_t indicator_color;
    if (ramUsage > 90) {
      indicator_color = lv_color_hex(0xff4757);  // 赤
    } else if (ramUsage > 75) {
      indicator_color = lv_color_hex(0xff6348);  // オレンジ赤
    } else if (ramUsage > 50) {
      indicator_color = lv_color_hex(0xffa502);  // オレンジ
    } else {
      indicator_color = lv_color_hex(0x4fc3f7);  // 青
    }
    
    lv_obj_set_style_bg_color(ram_bar, indicator_color, LV_PART_INDICATOR);
    lv_obj_set_style_text_color(ram_percent, indicator_color, 0);

    // RAM詳細情報の更新（見やすい形式）
    char ramDetailText[100];
    snprintf(ramDetailText, sizeof(ramDetailText),
             "Used: %.2fMB  |  Free: %.2fMB  |  Total: %.2fMB",
             usedMemoryMB, freeMemoryMB, totalMemoryMB);
    lv_label_set_text(ram_detail, ramDetailText);

    // ヒープ詳細情報の更新（見やすい形式）
    char heapInfo[140];
    snprintf(heapInfo, sizeof(heapInfo),
             "Min Free: %luKB  |  Max Alloc: %luKB\nCurrent Free: %luKB",
             (unsigned long)(minFreeRAM / 1024),
             (unsigned long)(maxAllocRAM / 1024),
             (unsigned long)(freeRAM / 1024));
    lv_label_set_text(heap_info_label, heapInfo);
  }

  // システム情報の更新（見やすい形式）
  unsigned long currentTime = millis();
  unsigned long uptimeSeconds = currentTime / 1000;
  unsigned long uptimeMinutes = uptimeSeconds / 60;
  unsigned long uptimeHours = uptimeMinutes / 60;
  
  uptimeSeconds %= 60;
  uptimeMinutes %= 60;

  uint32_t cpuFreq = getCpuFrequencyMhz();
  bool wifiConnected = WiFi.status() == WL_CONNECTED;
  uint32_t freeStack = uxTaskGetStackHighWaterMark(NULL);
  
  snprintf(shared_text_buffer, sizeof(shared_text_buffer),
           "CPU: %luMHz  |  WiFi: %s  |  Stack: %luB\nUptime: %02lu:%02lu:%02lu",
           cpuFreq, wifiConnected ? "Connected" : "Disconnected", (unsigned long)freeStack,
           uptimeHours, uptimeMinutes, uptimeSeconds);

  lv_label_set_text((lv_obj_t *)current_canvas, shared_text_buffer);
}




// Create screen 5 - System Data Display (Improved Design)
void create_screen5() {
  // 既存の画面を破棄
  destroy_current_screen();

  current_screen_obj = lv_obj_create(NULL);

  // 背景色設定（ダークブルーグラデーション風）
  lv_obj_remove_style_all(current_screen_obj);
  lv_obj_set_style_bg_color(current_screen_obj, lv_color_make(0x0A, 0x0A, 0x20), 0);
  lv_obj_set_style_bg_opa(current_screen_obj, LV_OPA_COVER, 0);
  
  // メイン画面のスクロールを完全に無効化
  lv_obj_set_scrollbar_mode(current_screen_obj, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(current_screen_obj, LV_OBJ_FLAG_SCROLLABLE);

  // ヘッダー背景
  lv_obj_t *header_bg = lv_obj_create(current_screen_obj);
  lv_obj_set_size(header_bg, 320, 35);
  lv_obj_align(header_bg, LV_ALIGN_TOP_MID, 0, 0);
  lv_obj_set_style_bg_color(header_bg, lv_color_make(0x15, 0x15, 0x35), 0);
  lv_obj_set_style_bg_opa(header_bg, LV_OPA_COVER, 0);
  lv_obj_set_style_border_width(header_bg, 0, 0);
  lv_obj_set_style_radius(header_bg, 0, 0);
  
  // ヘッダーのスクロールを無効化
  lv_obj_set_scrollbar_mode(header_bg, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(header_bg, LV_OBJ_FLAG_SCROLLABLE);

  // タイトルラベル
  lv_obj_t *title_label = lv_label_create(header_bg);
  lv_label_set_text(title_label, "SYSTEM INFORMATION");
  lv_obj_align(title_label, LV_ALIGN_CENTER, 0, 0);
  lv_obj_set_style_text_color(title_label, lv_color_make(0x00, 0xD4, 0xFF), 0); // シアン
  lv_obj_set_style_text_font(title_label, &lv_font_montserrat_14, 0);

  // メインコンテンツエリア
  lv_obj_t *content_area = lv_obj_create(current_screen_obj);
  lv_obj_set_size(content_area, 310, 195);
  lv_obj_align(content_area, LV_ALIGN_CENTER, 0, 8);
  lv_obj_set_style_bg_color(content_area, lv_color_make(0x12, 0x12, 0x28), 0);
  lv_obj_set_style_bg_opa(content_area, LV_OPA_COVER, 0);
  lv_obj_set_style_border_color(content_area, lv_color_make(0x30, 0x30, 0x50), 0);
  lv_obj_set_style_border_width(content_area, 1, 0);
  lv_obj_set_style_radius(content_area, 8, 0);
  lv_obj_set_style_pad_all(content_area, 8, 0);
  
  // スクロール機能を完全に無効化
  lv_obj_set_scrollbar_mode(content_area, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(content_area, LV_OBJ_FLAG_SCROLLABLE);

  // 上部エリア（DEVICE & STATUS - 左上）
  lv_obj_t *top_left = lv_obj_create(content_area);
  lv_obj_set_size(top_left, 145, 85);
  lv_obj_align(top_left, LV_ALIGN_TOP_LEFT, 0, 0);
  lv_obj_set_style_bg_opa(top_left, LV_OPA_TRANSP, 0);
  lv_obj_set_style_border_width(top_left, 0, 0);
  lv_obj_set_style_pad_all(top_left, 0, 0);
  lv_obj_set_scrollbar_mode(top_left, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(top_left, LV_OBJ_FLAG_SCROLLABLE);

  // 上部エリア（HARDWARE - 右上）
  lv_obj_t *top_right = lv_obj_create(content_area);
  lv_obj_set_size(top_right, 145, 85);
  lv_obj_align(top_right, LV_ALIGN_TOP_RIGHT, 0, 0);
  lv_obj_set_style_bg_opa(top_right, LV_OPA_TRANSP, 0);
  lv_obj_set_style_border_width(top_right, 0, 0);
  lv_obj_set_style_pad_all(top_right, 0, 0);
  lv_obj_set_scrollbar_mode(top_right, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(top_right, LV_OBJ_FLAG_SCROLLABLE);

  // 下部エリア（NETWORK - 横全体）
  lv_obj_t *bottom_full = lv_obj_create(content_area);
  lv_obj_set_size(bottom_full, 295, 95);
  lv_obj_align(bottom_full, LV_ALIGN_BOTTOM_MID, 0, 0);
  lv_obj_set_style_bg_opa(bottom_full, LV_OPA_TRANSP, 0);
  lv_obj_set_style_border_width(bottom_full, 0, 0);
  lv_obj_set_style_pad_all(bottom_full, 0, 0);
  lv_obj_set_scrollbar_mode(bottom_full, LV_SCROLLBAR_MODE_OFF);
  lv_obj_clear_flag(bottom_full, LV_OBJ_FLAG_SCROLLABLE);

  // 左上エリア（DEVICE & STATUS）のラベル作成
  create_info_label(top_left, "DEVICE", 0, 0, true);
  sys_data_labels[0] = create_info_label(top_left, "", 0, 18, false);  // model
  sys_data_labels[1] = create_info_label(top_left, "", 0, 32, false);  // version
  
  create_info_label(top_left, "STATUS", 0, 55, true);
  sys_data_labels[5] = create_info_label(top_left, "", 0, 73, false); // update status

  // 右上エリア（HARDWARE）のラベル作成
  create_info_label(top_right, "HARDWARE", 0, 0, true);
  sys_data_labels[2] = create_info_label(top_right, "", 0, 18, false);  // sensor
  sys_data_labels[3] = create_info_label(top_right, "", 0, 32, false);  // brightness
  sys_data_labels[4] = create_info_label(top_right, "", 0, 46, false); // update speed

  // 下部エリア（NETWORK）のラベル作成
  create_info_label(bottom_full, "NETWORK", 0, 0, true);
  sys_data_labels[6] = create_info_label(bottom_full, "", 0, 18, false);  // ip
  sys_data_labels[7] = create_info_label(bottom_full, "", 0, 32, false);  // ssid
  sys_data_labels[8] = create_info_label(bottom_full, "", 0, 46, false);  // mac
  sys_data_labels[9] = create_info_label(bottom_full, "", 0, 60, false);  // uuid

  // current_canvasにコンテンツエリアを設定
  current_canvas = content_area;

  // 初期表示
  update_system_data_display();

  Serial.println("Screen 5 created successfully");
}

// ヘルパー関数：情報ラベル作成
lv_obj_t* create_info_label(lv_obj_t *parent, const char* text, int x, int y, bool is_header) {
  lv_obj_t *label = lv_label_create(parent);
  lv_label_set_text(label, text);
  lv_obj_align(label, LV_ALIGN_TOP_LEFT, x, y);
  
  if (is_header) {
    lv_obj_set_style_text_color(label, lv_color_make(0xFF, 0xA5, 0x00), 0); // オレンジ
    lv_obj_set_style_text_font(label, &lv_font_montserrat_12, 0);
  } else {
    lv_obj_set_style_text_color(label, lv_color_white(), 0);
    lv_obj_set_style_text_font(label, &lv_font_montserrat_10, 0);
  }
  
  return label;
}

// システムデータ表示更新関数（改良版）
void update_system_data_display() {
  if (current_canvas == nullptr) return;

  // ラベルが初期化されているか確認
  if (sys_data_labels[0] == nullptr) return;

  // システム情報を取得
  String ip = SukenWiFi.getLocalIP();
  String ssid = SukenWiFi.getConnectedSSID();
  String mac = SukenWiFi.getDeviceMAC();

  // 各ラベルを個別に更新
  // デバイス情報
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "Model: %s", model.c_str());
  lv_label_set_text(sys_data_labels[0], shared_text_buffer);

  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "Ver: %d", Version);
  lv_label_set_text(sys_data_labels[1], shared_text_buffer);

  // ハードウェア情報  
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "Sensor: v%d", env_sensor_ver);
  lv_label_set_text(sys_data_labels[2], shared_text_buffer);

  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "LCD: %d%%", LCDbrightness);
  lv_label_set_text(sys_data_labels[3], shared_text_buffer);

  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "Refresh: %dms", UpdateSpeed);
  lv_label_set_text(sys_data_labels[4], shared_text_buffer);

  // ステータス
  lv_label_set_text(sys_data_labels[5], updatedNeeded ? "Update Ready" : "Up to Date");
  if (updatedNeeded) {
    lv_obj_set_style_text_color(sys_data_labels[5], lv_color_make(0xFF, 0x66, 0x66), 0); // 赤
  } else {
    lv_obj_set_style_text_color(sys_data_labels[5], lv_color_make(0x66, 0xFF, 0x66), 0); // 緑
  }

  // ネットワーク情報（MACとUUIDは短縮せず）
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "IP: %s", ip.c_str());
  lv_label_set_text(sys_data_labels[6], shared_text_buffer);

  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "SSID: %s", ssid.c_str());
  lv_label_set_text(sys_data_labels[7], shared_text_buffer);

  // MAC（フル表示）
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "MAC: %s", mac.c_str());
  lv_label_set_text(sys_data_labels[8], shared_text_buffer);

  // UUID（フル表示）
  snprintf(shared_text_buffer, sizeof(shared_text_buffer), "UUID: %s", uuid.c_str());
  lv_label_set_text(sys_data_labels[9], shared_text_buffer);
}

// 画面を破棄する関数（改善版）
void destroy_current_screen() {
  if (current_screen_obj != nullptr) {
    // 全ての子オブジェクトを削除
    lv_obj_clean(current_screen_obj);
    // オブジェクト自体を削除
    lv_obj_del(current_screen_obj);
    current_screen_obj = nullptr;
    current_canvas = nullptr;

    // LVGLのメモリを整理

    Serial.println("Screen destroyed");
  }
}









// TaskSchedulerインスタンス
Scheduler taskScheduler;


// タスクオブジェクトの宣言をここに移動（グローバル変数として）
Task tGetSensor(250, TASK_FOREVER, &getSensor, &taskScheduler);       // 1秒間隔
Task tUploadData(500, TASK_FOREVER, &uploadData, &taskScheduler);     // 5秒間隔
Task tGetSetting(30000, TASK_FOREVER, &getSetting, &taskScheduler);    // 30秒間隔
Task tSaveSD(10000, TASK_FOREVER, &SaveSD, &taskScheduler);            // 10秒間隔
Task tUpdateCheck(30000, TASK_FOREVER, &UpdateCheck, &taskScheduler);  // 30秒間隔





////////////////////////////////////////////////////////////////////////////////
//FreeRTOSイベント駆動型状態管理システム
////////////////////////////////////////////////////////////////////////////////




// ==== ONLINE_MODE ====
void enterONLINE_MODE() {
  // lcd.println("OnlineMode");  // LVGLに統一のためコメントアウト
  esp32FOTA.setManifestURL(manifest_url);
  esp32FOTA.printConfig();
//  SukenWiFi.onEnterSetupMode(displayAPQRCode);
//  SukenWiFi.onClientConnect(displaySettingPageQR);
  SukenWiFi.initBlocking(model);

  if (!SPIFFS.exists("/uuid.txt")) {
    getUUID();
  }

  // NTPとRTCの設定
  configTime(JST, 0, "ntp.nict.jp", "ntp.jst.mfeed.ad.jp");
  setNTP2RTC();
  readUUID();

  tGetSensor.enable();
  tUploadData.enable();
  tGetSetting.enable();
  //tUpdateCheck.enable();

  xTaskCreatePinnedToCore(WebServerTask, "WebServerTask", 8192, NULL, 2, &thp[2], 0);
}


void loopONLINE_MODE() {
  // Handle button presses for screen transitions
  handle_screen_transition();
  // 時計表示を定期更新（screen2が表示中の場合のみ）
  static unsigned long last_clock_update = 0;
  if (current_screen == 1 && millis() - last_clock_update > 400) {  // 1秒ごと
    update_clock_display();
    last_clock_update = millis();
  }

  // センサー表示を定期更新（screen3が表示中の場合のみ）
  static unsigned long last_sensor_update = 0;
  if (current_screen == 2 && millis() - last_sensor_update > 400) {  // 1秒ごと
    update_sensor_display();
    last_sensor_update = millis();
  }
  static unsigned long last_system_update = 0;
  if (current_screen == 3 && millis() - last_system_update > 400) {  // 2秒ごと
    update_system_display();
    last_system_update = millis();
  }
  // システムデータを定期更新（screen5が表示中の場合のみ）
  static unsigned long last_sysdata_update = 0;
  if (current_screen == 4 && millis() - last_sysdata_update > 400) {  // 5秒ごと
    update_system_data_display();
    last_sysdata_update = millis();
  }
  lv_timer_handler(); /* let the GUI do its work */
  delay(5);           /* let this time pass */
}
void exitONLINE_MODE() {
  tGetSensor.disable();
  tUploadData.disable();
  tGetSetting.disable();
  tUpdateCheck.disable();
}

// OFFLINE
void enterOFFLINE_MODE() {
  // SDカードの初期化
  if (!SD.begin(SD_CS_PIN)) {
    Serial.println("SD initialization failed!");
    return;
  }

  // data.txtファイルの作成
  File dataFile = SD.open("/data.txt", FILE_APPEND);
  if (!dataFile) {
    Serial.println("Failed to open data.txt");
    return;
  }
  dataFile.close();

  // オフラインモード用タスクを作成
  tGetSensor.enable();
  tSaveSD.enable();

  APStart();
  // lcd.println("OfflineMode");  // LVGLに統一のためコメントアウト
}


void loopOFFLINE_MODE() {
  Serial.println("OFFLINE_MODE loop");
  delay(500);
}

void exitOFFLINE_MODE() {
  tGetSensor.disable();
  tSaveSD.disable();
}


// OTA
void enterOTA() {
  Serial.println("OTA start");
}

// ERROR_OCCURRED
void enterERROR_OCCURRED() {
  Serial.println("ERROR_OCCURRED start");
}




struct State {
  void (*onEnter)();
  void (*onLoop)();
  void (*onExit)();
};
State states[] = {
#define X(name, loop, enter, exit) { enter, loop, exit },
  STATE_LIST
#undef X
};
const char *stateName(DeviceState s) {
  switch (s) {
#define X(name, loop, enter, exit) \
  case name: return #name;
    STATE_LIST
#undef X
    default: return "UNKNOWN";
  }
}

DeviceState currentState = STATE_MAX;
DeviceState previousState = STATE_MAX;

inline void changeState(DeviceState newState) {
  if (newState == currentState) return;

  if (currentState < STATE_MAX) {
    Serial.print("Exiting state: ");
    Serial.println(stateName(currentState));
    if (states[currentState].onExit != nullptr) states[currentState].onExit();
  }

  previousState = currentState;
  currentState = newState;

  Serial.print("Entering state: ");
  Serial.println(stateName(currentState));
  if (states[currentState].onEnter != nullptr) states[currentState].onEnter();
}



////////////////////////////////////////////////////////////////////////////////
//最初の処理
////////////////////////////////////////////////////////////////////////////////
void setup() {
  // 基本初期化
  M5.begin();
  Serial.begin(115200);
  Serial.println(motd);
  Wire.begin();
  SPIFFS.begin();

  // M5Displayの明示的な初期化
  M5.Display.begin();
  M5.Display.setRotation(1);
  M5.Display.setBrightness(100);
  M5.Display.fillScreen(TFT_BLACK);

  lv_init();
  lv_tick_set_cb(my_tick);

#if LV_USE_LOG != 0
  lv_log_register_print_cb(my_print);
#endif

  lv_display_t *disp;
  disp = lv_display_create(TFT_HOR_RES, TFT_VER_RES);
  lv_display_set_flush_cb(disp, my_disp_flush);
  lv_display_set_buffers(disp, draw_buf, nullptr, sizeof(draw_buf), LV_DISPLAY_RENDER_MODE_PARTIAL);

  /*Initialize the (dummy) input device driver*/
  lv_indev_t *indev = lv_indev_create();
  lv_indev_set_type(indev, LV_INDEV_TYPE_POINTER); /*Touchpad should have POINTER type*/
  lv_indev_set_read_cb(indev, my_touchpad_read);

  Serial.println("LVGL Display buffer initialized");

  delay(100);

  create_screen1();
  lv_screen_load(current_screen_obj);
  current_screen = 0;

  Serial.println("Screen 1 loaded");

  // センサー検出とタスク初期化
  sensor_detect();

  if (SD.begin(SD_CS_PIN, SPI, 40000000)) {
    changeState(OFFLINE_MODE);
  } else {
    changeState(ONLINE_MODE);
  }

  Serial.println("起動しました");
}

////////////////////////////////////////////////////////////////////////////////
//センサー関連
////////////////////////////////////////////////////////////////////////////////

void sensor_detect() {
  // I2Cデバイス(0x44)の検知（SHT40）
  Wire.beginTransmission(0x44);
  byte error = Wire.endTransmission();

  // 結果に応じてenv_sensor_verを設定
  if (error == 0) {
    env_sensor_ver = 4;
    Serial.println("SHT40 sensor detected");
  } else {
    env_sensor_ver = 1;
    Serial.println("SHT40 sensor not found");
  }

  // BMP280の検知（0x76または0x77）
  Wire.beginTransmission(0x76);
  error = Wire.endTransmission();
  if (error == 0) {
    Serial.println("BMP280 sensor detected at 0x76");
  } else {
    Wire.beginTransmission(0x77);
    error = Wire.endTransmission();
    if (error == 0) {
      Serial.println("BMP280 sensor detected at 0x77");
    } else {
      Serial.println("BMP280 sensor not found");
    }
  }

  // 結果をシリアルモニタに出力
  Serial.print("Env sensor version: ");
  Serial.println(env_sensor_ver);
}



////////////////////////////////////////////////////////////////////////////////
//メインのループです
////////////////////////////////////////////////////////////////////////////////
void loop() {   // メインのループ
  M5.update();  //マイコンの状態を更新
  // TaskSchedulerでタスクを実行
  taskScheduler.execute();
  if (states[currentState].onLoop != nullptr) {
    states[currentState].onLoop();
  }
}  //void loop終了

////////////////////////////////////////////////////////////////////////////////
//イベントハンドラー関数
////////////////////////////////////////////////////////////////////////////////

void displayWiFiError() {
  // LVGLに統一のためコメントアウト
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_RED);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("WiFi接続エラー");
  // canvas.println("再接続を試行中...");
  // canvas.pushSprite(0, 0);
}



void handleOfflineMode() {
  // オフラインモード用の処理（LVGLに統一のためコメントアウト）
  // lcd.println("OfflineMode - SD Card");
  delay(1000);
}

void handleReset() {
  // リセット処理（LVGLに統一のためコメントアウト）
  // lcd.println("Reset Requested");
  delay(1000);
  ESP.restart();
}

void handleError() {
  // エラー処理（LVGLに統一のためコメントアウト）
  // lcd.println("Error Occurred");
  delay(1000);
}



void APStart() {
  Serial.println("APスタート");  //WiFiを確認しシリアルに送信(開発用)
  WiFi.mode(WIFI_AP);
  WiFi.softAP(model.c_str());
  delay(200);  // Important! This delay is necessary
  WiFi.softAPConfig("192.168.1.100", "192.168.1.100", IPAddress(255, 255, 255, 0));
  xTaskCreatePinnedToCore(WebServerTask, "WebServerTask", 8192, NULL, 2, &thp[2], 1);
}


void handleRoot() {
  server.send(200, "text/html", root_html());
}

void AboutUS() {
  server.send(200, "text/html", aboutus_html());
}

void UpdateAPI() {
  JsonDocument doc;
  doc["updatedNeeded"] = updatedNeeded;
  String jsonPayload;
  serializeJson(doc, jsonPayload);
  server.send(200, "application/json", jsonPayload);
}

void sensorapi() {
  delay(100);
  M5.Rtc.getDateTime(&RTCDateTime);  // 現在日時を取得
  String Date = String(RTCDateTime.date.year) + "/" + String(RTCDateTime.date.month) + "/" + String(RTCDateTime.date.date);
  String Time = String(RTCDateTime.time.hours) + ":" + String(RTCDateTime.time.minutes) + ":" + String(RTCDateTime.time.seconds);  // 現在時刻を表示
  JsonDocument doc;
  doc["BordModel"] = model;
  doc["SystemVersion"] = Version;
  doc["uuid"] = uuid;
  doc["MAC"] = SukenWiFi.getDeviceMAC();
  doc["tmp"] = tmp;
  doc["hum"] = hum;
  doc["pressure"] = pressure;
  doc["SoilSensor"] = SoilSensorVoltage;
  doc["BTLevel"] = BTLevel;
  doc["BTVoltage"] = BTVoltage;
  doc["Date"] = Date;
  doc["Time"] = Time;
  String jsonPayload;
  serializeJson(doc, jsonPayload);
  server.send(200, "application/json", jsonPayload);
}

void WebServerTask(void *args) {
  server.sendHeader("Access-Control-Allow-Origin", "*");
  server.sendHeader("Access-Control-Max-Age", "10000");
  server.sendHeader("Content-Length", "0");
  server.on("/api/sensor", sensorapi);
  server.on("/api/Update", UpdateAPI);
  server.begin();
  Serial.println("Webサーバー開始");
  while (1) {

    server.handleClient();
    a++;
    delay(100);
  }
}




//バッテリーチャージの設定falseで充電しない
void setBatteryCharge(bool enabled) {
  M5.Power.setBatteryCharge(enabled);
}


void btProtect() {
  if (BTLevel > 80) {
    setBatteryCharge(false);
    Serial.println("バッテリーの充電を停止");
  }
  if (BTLevel < 30) {
    setBatteryCharge(true);
    Serial.println("バッテリーの充電を開始");
  }
}

////////////////////////////////////////////////////////////////////////////////
//ボタン関係
////////////////////////////////////////////////////////////////////////////////
//ページ遷移ボタン
// Handle screen transitions
void handle_screen_transition() {
  static bool btnA_was_pressed = false;
  static bool btnB_was_pressed = false;

  bool btnA_pressed = M5.BtnA.isPressed();
  bool btnB_pressed = M5.BtnB.isPressed();

  // Button A - Previous page
  if (btnA_pressed && !btnA_was_pressed) {
    current_screen--;
    if (current_screen < 0) {
      current_screen = 4;  // 4画面なので0-3
    }
    load_current_screen();
  }

  // Button B - Next page
  if (btnB_pressed && !btnB_was_pressed) {
    current_screen++;
    if (current_screen > 4) {  // 4画面なので0-3
      current_screen = 0;
    }
    load_current_screen();
  }

  btnA_was_pressed = btnA_pressed;
  btnB_was_pressed = btnB_pressed;
}
void load_current_screen() {
  switch (current_screen) {
    case 0:
      create_screen1();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 1");
      break;
    case 1:
      create_screen2();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 2");
      break;
    case 2:
      create_screen3();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 3");
      break;
    case 3:
      create_screen4();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 4");
      break;
    case 4:
      create_screen5();
      lv_screen_load(current_screen_obj);
      Serial.println("Switched to Screen 5");
      break;
  }
}



//リセットの処理
void ResetButton(void *args) {
  while (1) {
    if (M5.BtnC.isPressed()) {  //ボタンの感知
      resetbutton += 1;         //ボタンが押されている場合は値が増加します
    } else {                    //if文の「または~」
      if (resetbutton > 0) {    //値が0を下回らないように
        resetbutton -= 1;       //ボタンが押されていない場合は値が減少します
      }                         //if文終了
    }                           //if文終了

    // lcd.pushImage(0, 0, imgWidth, imgHeight, reset);  //警告画面を表示（LVGLに統一のためコメントアウト）
    // lcd.endWrite();                                   //描画を一度終了します（LVGLに統一のためコメントアウト）


    Serial.println("リセットモードに入りました");  //ステータスを通知                                              //if文終了

    if (resetbutton > 800) {  //リセットボタンの値が一定数増えたら実行
      // lcd.setTextSize(2);                      // 文字サイズを設定（LVGLに統一のためコメントアウト）
      // lcd.setCursor(0, 0);                     //カーソルの設定（LVGLに統一のためコメントアウト）
      // lcd.fillScreen(TFT_BLACK);               //黒で塗りつぶす（LVGLに統一のためコメントアウト）
      // lcd.println("Resetting");                //文字列を表示（LVGLに統一のためコメントアウト）
      // lcd.println("It may take some time");    //文字列を表示氏（LVGLに統一のためコメントアウト）
      // lcd.println("The system will restart");  //文字列を表示（LVGLに統一のためコメントアウト）
      SPIFFS.format();        //内部データを削除
      ESP.restart();          //再起動
    }                         //if文終了
    if (resetbutton < 290) {  //リセットボタンの値が一定数増えたら実行
      // リセットモードを解除し、前の状態に戻す

      page_transition = 1;                             //ページ遷移モードを有効に
      Serial.println("リセットモードを解除しました");  //シリアルで解除を通知
    }                                                  //if文終了
    delay(100);
    a++;
  }  //if文終了
}

















////////////////////////////////////////////////////////////////////////////////
// 設定関連
////////////////////////////////////////////////////////////////////////////////
void displaySettingPageQR() {
  // LVGLに統一のためコメントアウト
  // String QR = "http://192.168.1.100";
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_BLACK);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("下記のリンクにアクセスしてください");
  // canvas.qrcode(QR, 50, 40, 190, 5);
  // canvas.pushSprite(0, 0);  //描画を一度終了します
}


void displayAPQRCode() {
  changeState(SETTING);
  // LVGLに統一のためコメントアウト
  // String QR = "WIFI:S:" + model + ";T:nopass;P:;H:false;;";
  // canvas.setCursor(0, 0);
  // canvas.fillScreen(TFT_BLACK);
  // canvas.setFont(&fonts::Font2);
  // canvas.println("QRコードを読み込んでWiFiに接続");
  // canvas.qrcode(QR, 50, 40, 190, 5);
  // canvas.pushSprite(0, 0);
}


//UUIDを取得
void getUUID() {
  HTTPClient http;                                      //httpクライアントを定義
  http.begin("https://" + host + "/M5API/getuid.php");  //サーバーのアドレス
  int httpCode = http.GET();                            //サーバーに接続
  // HTTP応答の処理
  if (httpCode > 0) {                                      //内容があるか確認
    String payload = http.getString();                     //変数にGETの内容代入
    Serial.println("HTTP GET successful");                 //シリアルで通信成功を通知(開発用)
    File file = SPIFFS.open("/uuid.txt", "w");             // SPIFFSに書き込み
    if (file) {                                            //ファイルチェック
      file.print(payload);                                 //ファイルに書き込み
      file.close();                                        //ファイルを閉じる
      Serial.println("Data written to SPIFFS");            //シリアルで書き込み成功を通知(開発用)
      ESP.restart();                                       //再起動
    } else {                                               //if文「または」
      Serial.println("Failed to open file for writing");   //失敗をシリアルで通知
    }                                                      //if文終了
  } else {                                                 //if文「または」
    Serial.print("HTTP GET failed with error code: ");     //シリアルでエラーを通知(開発用)
    Serial.println("HTTP GET failed");                     //シリアルでエラーを通知(開発用)
    Serial.println(http.errorToString(httpCode).c_str());  //シリアルでエラーを通知(開発用)
  }                                                        //if文終了
  http.end();                                              //httpを終了する
}  //void getUUID終了




//RTCに書き込み
//RTCに書き込み
void debugRTCInfo() {
  Serial.println("=== RTC デバッグ情報 ===");

  // M5Stackの機種確認
  Serial.printf("M5Stack機種: %s\n", M5.getBoard());

  // RTC設定前の現在値を確認
  Serial.println("--- RTC設定前の値 ---");
  M5.Rtc.getDateTime(&RTCDateTime);
  Serial.printf("年: %d, 月: %d, 日: %d\n",
                RTCDateTime.date.year,
                RTCDateTime.date.month,
                RTCDateTime.date.date);
  Serial.printf("時: %d, 分: %d, 秒: %d, 曜日: %d\n",
                RTCDateTime.time.hours,
                RTCDateTime.time.minutes,
                RTCDateTime.time.seconds,
                RTCDateTime.date.weekDay);

  // Raw値も確認（もし利用可能なら）
  Serial.printf("Raw RTC値: 年=%d, 月=%d, 日=%d\n",
                RTCDateTime.date.year,
                RTCDateTime.date.month,
                RTCDateTime.date.date);
}

void setNTP2RTC() {
  configTime(JST, 0, "ntp.nict.jp", "ntp.jst.mfeed.ad.jp");  //NTPサーバーの設定
  while (!getLocalTime(&timeinfo)) {                         //繰り返し
    delay(100);                                              //一秒待つ
  }
  getLocalTime(&timeinfo);  //RTC関係
  // RTCを更新する
  M5.Rtc.getDateTime(&RTCDateTime);
  RTCDateTime.date.year = timeinfo.tm_year + 1900;
  RTCDateTime.date.month = timeinfo.tm_mon + 1;
  RTCDateTime.date.date = timeinfo.tm_mday;
  RTCDateTime.date.weekDay = timeinfo.tm_wday;
  RTCDateTime.time.hours = timeinfo.tm_hour;
  RTCDateTime.time.minutes = timeinfo.tm_min;
  RTCDateTime.time.seconds = timeinfo.tm_sec;
  M5.Rtc.setDateTime(&RTCDateTime);
  Serial.println("RTCを設定しました");  //シリアルでRTC設定完了を通知(開発用)
}


//UUIDを読み込みます
void readUUID() {
  File file = SPIFFS.open("/uuid.txt", "r");            //uuid.txtを開く
  if (!file) {                                          //ファイルチェック
    Serial.println("uuid.txt ファイルが開けません！");  //シリアルで通知(開発用)
    return;                                             //リターン
  }                                                     //if文終了
  uuid = file.readString();                             // ファイルの中身を読み取り、変数に代入
  file.close();                                         //ファイルを閉じる
  Serial.println("読み取ったUUID: " + uuid);            // 読み取ったUUIDをシリアルモニタに表示
}  //void readUUID終了













////////////////////////////////////////////////////////////////////////////////
// オンラインアップデート機能
////////////////////////////////////////////////////////////////////////////////
void getCert() {
  SPIFFS.remove("/root_ca.pem");                                     //root_ca.pemを削除
  HTTPClient http;                                                   //httpクライアントを定義
  http.begin("https://" + host + "/M5API/getCert.php?url=" + host);  //サーバーのアドレス
  int httpCode = http.GET();                                         //サーバーに接続
  if (httpCode > 0) {                                                //内容があるか確認
    String payload = http.getString();                               //変数にGETの内容代入
    Serial.println("HTTP GET successful");                           //シリアルで通信成功を通知(開発用)
    File file = SPIFFS.open("/root_ca.pem", "w");                    // SPIFFSに書き込み
    if (file) {                                                      //ファイルチェック
      file.print(payload);                                           //ファイルに書き込み
      file.close();                                                  //ファイルを閉じる
      Serial.println("Data written to SPIFFS");                      //シリアルで書き込み成功を通知(開発用)
      Serial.println(payload);                                       //シリアルで書き込み成功を通知(開発用)
    } else {                                                         //if文「または」
      Serial.println("Failed to open file for writing");             //失敗をシリアルで通知
    }                                                                //if文終了
  } else {                                                           //if文「または」
    Serial.print("HTTP GET failed with error code: ");               //シリアルでエラーを通知(開発用)
    Serial.println("HTTP GET failed");                               //シリアルでエラーを通知(開発用)
    Serial.println(http.errorToString(httpCode).c_str());            //シリアルでエラーを通知(開発用)
  }                                                                  //if文終了
  http.end();                                                        //httpを終了する
}


void ota() {
  if (updatedNeeded) {
    esp32FOTA.execOTA();
  }
  delay(2000);
}









////////////////////////////////////////////////////////////////////////////////
//jobs
////////////////////////////////////////////////////////////////////////////////
void getSensor() {  //センサーの値を取得する
  // SHT40から温度・湿度を取得
  if (env_sensor_ver == 4) {
    // SHT40の温度・湿度読み取り
    Wire.beginTransmission(0x44);
    Wire.write(0xFD);                   // 高精度測定コマンド
    if (Wire.endTransmission() == 0) {  // 送信成功を確認
      delay(10);

      Wire.requestFrom(0x44, 6);
      if (Wire.available() >= 6) {
        uint16_t temp_raw = (Wire.read() << 8) | Wire.read();
        Wire.read();  // temp CRC (読み飛ばし)
        uint16_t hum_raw = (Wire.read() << 8) | Wire.read();
        Wire.read();  // hum CRC (読み飛ばし)

        // SHT40の正しい変換式
        tmp = -45.0 + 175.0 * temp_raw / 65535.0;
        hum = -6.0 + 125.0 * hum_raw / 65535.0;

        // 範囲チェック
        if (hum < 0) hum = 0;
        if (hum > 100) hum = 100;
      }
    }
  }

  // BMP280から気圧を取得
  Wire.beginTransmission(0x76);
  Wire.write(0xF7);                   // 気圧データレジスタ
  if (Wire.endTransmission() == 0) {  // 送信成功を確認
    Wire.requestFrom(0x76, 3);
    if (Wire.available() >= 3) {
      uint32_t pressure_raw = ((uint32_t)Wire.read() << 12) | ((uint32_t)Wire.read() << 4) | (Wire.read() >> 4);

      // BMP280の場合、キャリブレーションが必要
      // 簡易的な変換（実際はキャリブレーション値を使用）
      pressure = pressure_raw / 256.0;  // 仮の変換式

      // 妥当な範囲チェック（300-1100hPa）
      if (pressure < 300 || pressure > 1100) {
        pressure = 1013.25;  // デフォルト値
      }
    }
  }

  // バッテリー情報取得
  BTVoltage = M5.Power.getBatteryVoltage() / 1000.0;  // mVからVに変換
  BTLevel = M5.Power.getBatteryLevel();               // バッテリー残量を取得

  // 実電圧計算（校正係数適用）
  SoilSensorVoltage = alpha * analogRead(CH1_PIN) * MAX_VOLTAGE / ANALOG_MAX;
}  //void getSensor終了

void uploadData() {
  HTTPClient http;
  JsonDocument doc;
  doc["BordModel"] = model;
  doc["SystemVersion"] = Version;
  doc["uuid"] = uuid;
  doc["tmp"] = tmp;
  doc["hum"] = hum;
  doc["pressure"] = pressure;
  doc["SoilSensor"] = SoilSensorVoltage;
  doc["BTLevel"] = BTLevel;
  doc["BTVoltage"] = BTVoltage;
  
  String jsonPayload;
  serializeJson(doc, jsonPayload);


  checkDNS(host.c_str());

  http.setReuse(false);
  Serial.println(jsonPayload);
  http.addHeader("Connection", "close");

  // WiFiClientSecure を渡して HTTPS 通信
  if (http.begin("https://" + host + "/M5API/postData.php")) {
    http.addHeader("Content-Type", "application/json");
    int httpResponseCode = http.POST(jsonPayload);

    if (httpResponseCode > 0) {
      Serial.print("HTTP Response code: ");
      Serial.println(httpResponseCode);
      String response = http.getString();
      Serial.println(response);
    } else {
      Serial.print("HTTP Request failed. Error code: ");
      Serial.println(httpResponseCode);
    }
    http.end();
  } else {
    Serial.println("HTTP begin() failed");
  }
}

void checkDNS(const char *hostname) {
  IPAddress ip;

  Serial.printf("Resolving: %s\n", hostname);

  if (WiFi.hostByName(hostname, ip)) {
    Serial.printf("DNS Success: %s -> %s\n", hostname, ip.toString().c_str());
  } else {
    Serial.printf("DNS Failed: %s\n", hostname);
  }
}

void getSetting() {
  HTTPClient http;
  http.setReuse(false);
  http.addHeader("Connection", "close");
  http.begin("https://" + host + "/M5API/getSetting.php?uuid=" + uuid);
  int httpCode = http.GET();
  if (httpCode > 0) {
    // 成功したらJSONデータを取得する
    String payload = http.getString();
    Serial.println(payload);

    // JSONデータをパースして変数に代入する
    // 以下はArduinoJsonライブラリを使用した例
    JsonDocument doc;
    deserializeJson(doc, payload);

    // DisplayImgUrlの取得
    DisplayImgUrl = doc["DisplayImgUrl"].as<String>();
    Serial.println("DisplayImgUrl: " + DisplayImgUrl);

    // LCDbrightnessの取得
    LCDbrightness = doc["LCDbrightness"].as<int>();
    Serial.println("LCDbrightness: " + String(LCDbrightness));

    // UpdateSpeedの取得
    UpdateSpeed = doc["UpdateSpeed"].as<int>();
    Serial.println("UpdateSpeed: " + String(UpdateSpeed));

    // LEDの取得
    LED = doc["LED"].as<int>();
    Serial.println("LED: " + String(LED));

    // LEDの取得
    BTprotection = doc["BTprotection"].as<int>();
    Serial.println("BTprotection: " + String(BTprotection));

    SystemBackColor = doc["SystemBackColor"].as<int>();
    Serial.println("SystemBackColor: " + String(SystemBackColor));

    SystemTextColor = doc["SystemTextColor"].as<int>();
    Serial.println("SystemTextColor: " + String(SystemTextColor));

    ClockBackColor = doc["ClockBackColor"].as<String>();
    Serial.println("ClockBackColor: " + ClockBackColor);

    ClockTextColor = doc["ClockTextColor"].as<String>();
    Serial.println("ClockTextColor: " + ClockTextColor);

  } else {
    Serial.println("Failed to connect to API");
  }

  http.end();
  // lcd.setBrightness(LCDbrightness);  // 画面の明るさを設定（LVGLに統一のためコメントアウト）
}  //void getSetting終了




void SaveSD() {
  M5.Rtc.getDateTime(&RTCDateTime);  // 現在日時を取得
  String Date = String(RTCDateTime.date.year) + "/" + String(RTCDateTime.date.month) + "/" + String(RTCDateTime.date.date);
  String Time = String(RTCDateTime.time.hours) + ":" + String(RTCDateTime.time.minutes) + ":" + String(RTCDateTime.time.seconds);  // 現在時刻を表示
  JsonDocument doc;
  doc["BordModel"] = model;
  doc["SystemVersion"] = Version;
  doc["tmp"] = tmp;
  doc["hum"] = hum;
  doc["pressure"] = pressure;
  doc["SoilSensor"] = SoilSensorVoltage;
  doc["BTLevel"] = BTLevel;
  doc["BTVoltage"] = BTVoltage;
  doc["Date"] = Date;
  doc["Time"] = Time;

  // JSONを文字列に変換
  String output;
  serializeJson(doc, output);

  // SDカードにデータを書き込む
  File dataFile = SD.open("/data.txt", FILE_APPEND);
  if (dataFile) {
    dataFile.println(output);
    dataFile.close();
    Serial.println("SDカードにセーブしました");
  } else {
    Serial.println("Failed to open data.txt");
  }
}



void UpdateCheck() {
  updatedNeeded = esp32FOTA.execHTTPcheck();
}